/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.principalusecount;

import java.util.Map;
import com.openexchange.exception.OXException;
import com.openexchange.session.Session;

/**
 * {@link PrincipalUseCountService} is a use-count service for objects with identifiers generated by the principal identifier generator.
 *
 * @author <a href="mailto:kevin.ruthmann@open-xchange.com">Kevin Ruthmann</a>
 * @since v7.10.2
 */
public interface PrincipalUseCountService {

    /**
     * Increments the use-count for the given principal object
     *
     * @param session The session of the current user
     * @param principal The id of the principal
     * @throws OXException
     */
    default void increment(Session session, int principal) throws OXException {
        increment(session, Args.builderForIncrement(principal).build());
    }

    /**
     * Increments the use-count for the given principal object
     *
     * @param session The session of the current user
     * @param args The arguments
     * @throws OXException
     */
    void increment(Session session, Args args) throws OXException;

    /**
     * Resets the use-count to <code>0</code> (zero) for the given principal object
     *
     * @param session The session of the current user
     * @param principal The id of the principal
     * @throws OXException
     */
    default void reset(Session session, int principal) throws OXException {
        reset(session, Args.builderForReset(principal).build());
    }

    /**
     * Resets the use-count to <code>0</code> (zero) for the given principal object
     *
     * @param session The session of the current user
     * @param args The arguments
     * @throws OXException
     */
    void reset(Session session, Args args) throws OXException;

    /**
     * Sets the use-count of the principal to the given value.
     *
     * @param session The session of the current user
     * @param principal The id of the principal
     * @param value The use-count value
     * @throws OXException
     */
    default void set(Session session, int principal, int value) throws OXException {
        set(session, Args.builderForSet(principal, value).build());
    }

    /**
     * Sets the use-count of the principal to the given value.
     *
     * @param session The session of the current user
     * @param args The arguments
     * @throws OXException
     */
    void set(Session session, Args args) throws OXException;

    /**
     * Returns a map of principal identifiers mapped to their use counts.
     *
     * @param session The user session
     * @param principals The principal identifiers
     * @return A map of principal identifiers mapped to their use counts
     * @throws OXException
     */
    Map<Integer, Integer> get(Session session, Integer... principals) throws OXException;

}
