/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.importexport.actions.exporter;

import static com.openexchange.java.Autoboxing.I;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import com.google.common.collect.ImmutableList;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.DispatcherNotes;
import com.openexchange.ajax.requesthandler.annotation.restricted.RestrictedAction;
import com.openexchange.exception.OXException;
import com.openexchange.folderstorage.ContentType;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.folderstorage.FolderStorage;
import com.openexchange.folderstorage.UserizedFolder;
import com.openexchange.groupware.container.FolderObject;
import com.openexchange.importexport.Exporter;
import com.openexchange.importexport.Format;
import com.openexchange.importexport.exporters.ICalExporter;
import com.openexchange.importexport.json.ExportRequest;
import com.openexchange.oauth.provider.resourceserver.OAuthAccess;
import com.openexchange.oauth.provider.resourceserver.annotations.OAuthScopeCheck;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.OAuthContentTypes;
import com.openexchange.tools.session.ServerSession;

@DispatcherNotes(defaultFormat = "file")
@RestrictedAction(module = AbstractExportAction.MODULE, type = RestrictedAction.Type.READ, hasCustomOAuthScopeCheck = true)
public class ICalExportAction extends AbstractExportAction {

    private static final List<Integer> ACCEPTED_OAUTH_MODULES = ImmutableList.of(I(FolderObject.CALENDAR), I(FolderObject.TASK));
    private final Exporter exporter;
    private final ServiceLookup services;

    /**
     * Initializes a new {@link ICalExportAction}.
     * 
     * @param services The service look-up
     */
    public ICalExportAction(ServiceLookup services) {
        super();
        this.services = services;
        exporter = new ICalExporter();
    }

    @Override
    public Format getFormat() {
        return Format.ICAL;
    }

    @Override
    public Exporter getExporter() {
        return exporter;
    }

    @OAuthScopeCheck
    public boolean accessAllowed(final AJAXRequestData request, final ServerSession session, final OAuthAccess access) throws OXException {
        ExportRequest exportRequest = new ExportRequest(request, session);
        Map<String, List<String>> batchIds = exportRequest.getBatchIds();
        if (false == isBatchExport(batchIds)) {
            return OAuthContentTypes.mayReadViaOAuthRequest(getContentTypeForUserizedFolder(exportRequest.getFolder(), session), access);
        }
        return OAuthContentTypes.mayReadViaOAuthRequest(collectContentTypes(batchIds, session), access);
    }

    /**
     * Collects the content types from the specified batch id map
     * 
     * @param batchIds the map
     * @param session The session
     * @return The content types
     * @throws OXException if an error is occurred
     */
    private Set<ContentType> collectContentTypes(Map<String, List<String>> batchIds, ServerSession session) throws OXException {
        short items = 0;
        Set<ContentType> contentTypes = new HashSet<>(4);
        for (String folderId : batchIds.keySet()) {
            if (false == contentTypes.add(getContentTypeForUserizedFolder(folderId, session))) {
                continue;
            }
            items++;
            if (items == ACCEPTED_OAUTH_MODULES.size()) {
                break;
            }
        }
        return contentTypes;
    }

    private boolean isBatchExport(Map<String, List<String>> batchIds) {
        return null != batchIds && false == batchIds.isEmpty();
    }

    /**
     * 
     * Gets the content type for a given folder id of a userized folder.
     *
     * @param folderId The folder id.
     * @param session The server session.
     * @return The content type.
     * @throws OXException If folder cannot be found.
     */
    protected ContentType getContentTypeForUserizedFolder(String folderId, final ServerSession session) throws OXException {
        FolderService folderService = services.getService(FolderService.class);
        UserizedFolder userizedFolder = folderService.getFolder(FolderStorage.REAL_TREE_ID, folderId, session, null);
        return userizedFolder.getContentType();
    }
}
