/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.html.internal.jsoup.control;

import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import com.openexchange.exception.OXException;
import com.openexchange.html.internal.jsoup.JsoupHandler;
import com.openexchange.html.internal.jsoup.JsoupParser;
import com.openexchange.threadpool.AbstractTask;
import com.openexchange.threadpool.ThreadRenamer;

/**
 * {@link JsoupParseTask}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.8.2
 */
public class JsoupParseTask extends AbstractTask<Void> implements Delayed {

    /** The special poison task to stop taking from queue */
    public static final JsoupParseTask POISON = new JsoupParseTask(null, null, 0, false) {

        @Override
        public int compareTo(Delayed o) {
            return -1;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return 0L;
        }
    };

    // ---------------------------------------------------------------------------------------------------------------------------

    private final String html;
    private final JsoupHandler handler;
    private final long timeoutMillis;
    private final boolean prettyPrint;
    private final AtomicLong stamp;
    private final AtomicReference<Thread> workerReference;

    /**
     * Initializes a new {@link JsoupParseTask}.
     *
     * @param html The HTML content to parse
     * @param handler The handler to call-back
     * @param timeoutSec The timeout seconds
     * @param prettyPrint Whether resulting HTML content is supposed to be pretty-printed
     */
    public JsoupParseTask(String html, JsoupHandler handler, int timeoutSec, boolean prettyPrint) {
        super();
        this.html = html;
        this.handler = handler;
        this.timeoutMillis = timeoutSec * 1000L;
        this.prettyPrint = prettyPrint;
        this.stamp = new AtomicLong(0);
        this.workerReference = new AtomicReference<Thread>(null);
    }

    @Override
    public int compareTo(Delayed o) {
        long thisStamp = this.stamp.get();
        long otherStamp = ((JsoupParseTask) o).stamp.get();
        return (thisStamp < otherStamp ? -1 : (thisStamp == otherStamp ? 0 : 1));
    }

    @Override
    public long getDelay(TimeUnit unit) {
        long toGo = stamp.get() - System.currentTimeMillis();
        return unit.convert(toGo, TimeUnit.MILLISECONDS);
    }

    @Override
    public void setThreadName(ThreadRenamer threadRenamer) {
        threadRenamer.renamePrefix("JsoupParser");
    }

    @Override
    public Void call() throws OXException {
        JsoupParseControl control = JsoupParseControl.getInstance();
        stamp.set(System.currentTimeMillis() + timeoutMillis);
        workerReference.set(Thread.currentThread());
        control.add(this);
        try {
            JsoupParser.doParse(html, handler, prettyPrint);
            return null;
        } finally {
            workerReference.set(null);
            control.remove(this);
        }
    }

    /**
     * Interrupts this task (if currently processed).
     */
    public void interrupt() {
        Thread worker = workerReference.get();
        if (null != worker) {
            worker.interrupt();
        }
    }

}
