/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.html.internal.jsoup;

import org.jsoup.nodes.Comment;
import org.jsoup.nodes.DataNode;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.DocumentType;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.TextNode;
import org.jsoup.nodes.XmlDeclaration;

/**
 * {@link JsoupHandler} - Receives call-backs on traversal of a parsed JSoup document.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.8.4
 */
public interface JsoupHandler {

    /**
     * Handles specified comment.
     *
     * @param comment The comment
     */
    void handleComment(Comment comment);

    /**
     * Handles specified data node.
     *
     * @param dataNode The data node
     */
    void handleDataNode(DataNode dataNode);

    /**
     * Handles specified document type.
     *
     * @param documentType The document type
     */
    void handleDocumentType(DocumentType documentType);

    /**
     * Handles specified text node.
     *
     * @param textNode The text node
     */
    void handleTextNode(TextNode textNode);

    /**
     * Handles specified XML declaration.
     *
     * @param xmlDeclaration The XML declaration
     */
    void handleXmlDeclaration(XmlDeclaration xmlDeclaration);

    /**
     * Handles start of specified element.
     *
     * @param element The element
     */
    void handleElementStart(Element element);

    /**
     * Handles end of specified element.
     *
     * @param element The element
     */
    void handleElementEnd(Element element);

    /**
     * Advertises finished parsing of specified document.
     *
     * @param document The document
     */
    void finished(Document document);

    /**
     * Gets the parsed JSoup document.
     *
     * @return The document
     */
    Document getDocument();

}
