/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.find.mail;

import static com.openexchange.java.util.Tools.getUnsignedInteger;
import java.util.LinkedList;
import java.util.List;
import com.openexchange.config.lean.LeanConfigurationService;
import com.openexchange.find.facet.FacetType;
import com.openexchange.find.osgi.Services;
import com.openexchange.java.Strings;
import com.openexchange.log.LogProperties;


/**
 * Facet types for the mail module.
 *
 * @author <a href="mailto:steffen.templin@open-xchange.com">Steffen Templin</a>
 * @since v7.6.0
 */
public enum MailFacetType implements FacetType {

    SUBJECT("subject"),
    MAIL_TEXT("mail_text"),
    CONTACTS("contacts", MailStrings.FACET_PEOPLE),
    FILENAME("filename"),
    HAS_ATTACHMENT("has_attachment", MailStrings.FACET_ATTACHMENT),
    CUSTOM_FLAG("custom_flag", MailStrings.FACET_CUSTOM_FLAG),
    ;

    private final String id;
    private final String displayName;
    private final List<FacetType> conflictingFacets = new LinkedList<FacetType>();

    private MailFacetType(String id) {
        this(id, null);
    }

    private MailFacetType(String id, String displayName) {
        this.id = id;
        this.displayName = displayName;
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public String getDisplayName() {
        if (CUSTOM_FLAG == this) {
            LeanConfigurationService configService = Services.optService(LeanConfigurationService.class);
            if (configService != null) {
                int contextId = getUnsignedInteger(LogProperties.get(LogProperties.Name.SESSION_CONTEXT_ID));
                if (contextId > 0) {
                    int userId = getUnsignedInteger(LogProperties.get(LogProperties.Name.SESSION_USER_ID));
                    if (userId > 0) {
                        return configService.getProperty(userId, contextId, MailFacetTypeProperty.CUSTOM_FLAG_DISPLAY_NAME);
                    }
                }
                return configService.getProperty(MailFacetTypeProperty.CUSTOM_FLAG_DISPLAY_NAME);
            }
        }
        return displayName;
    }

    @Override
    public List<FacetType> getConflictingFacets() {
        return conflictingFacets;
    }

    /**
     * Gets a {@link MailFacetType} by its identifier.
     *
     * @param id The identifier to look-up
     * @return The type or <code>null</code>, if the identifier is invalid.
     */
    public static MailFacetType getById(String id) {
        if (Strings.isEmpty(id)) {
            return null;
        }

        String toLookUp = Strings.asciiLowerCase(id);
        for (MailFacetType mailFacetType : MailFacetType.values()) {
            if (toLookUp.equals(mailFacetType.id)) {
                return mailFacetType;
            }
        }
        return null;
    }

}
