/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.contact.vcard.impl.mapping;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import com.openexchange.contact.vcard.VCardParameters;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.contact.helpers.ContactField;
import com.openexchange.groupware.container.Contact;
import com.openexchange.java.Strings;
import ezvcard.VCard;
import ezvcard.parameter.EmailType;
import ezvcard.property.Email;

/**
 * {@link EMailMapping}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 */
public class EMailMapping extends AbstractMapping {

    /**
     * Initializes a new {@link EMailMapping}.
     */
    public EMailMapping() {
        super("EMAIL", ContactField.EMAIL1, ContactField.EMAIL2, ContactField.EMAIL3);
    }

    @Override
    public void exportContact(Contact contact, VCard vCard, VCardParameters parameters, List<OXException> warnings) {
        MappedEmails mappedEmails = extractMappedEmails(vCard);
        /*
         * email1 - type "WORK"
         */
        if (has(contact, Contact.EMAIL1)) {
            boolean needsPrefType = 1 < getPropertiesWithTypes(vCard.getEmails(), EmailType.WORK.getValue()).size();
            if (null == mappedEmails.email1) {
                vCard.addEmail(contact.getEmail1(), needsPrefType ? new EmailType[] { EmailType.WORK, EmailType.PREF } : new EmailType[] { EmailType.WORK });
            } else {
                mappedEmails.email1.setValue(contact.getEmail1());
                addTypesIfMissing(mappedEmails.email1, needsPrefType ?
                    new String[] { EmailType.WORK.getValue(), EmailType.PREF.getValue() } : new String[] { EmailType.WORK.getValue() });
            }
        } else if (null != mappedEmails.email1) {
            vCard.removeProperty(mappedEmails.email1);
        }
        /*
         * email2 - type "HOME"
         */
        if (has(contact, Contact.EMAIL2)) {
            if (null == mappedEmails.email2) {
                vCard.addEmail(contact.getEmail2(), EmailType.HOME);
            } else {
                mappedEmails.email2.setValue(contact.getEmail2());
                addTypeIfMissing(mappedEmails.email2, EmailType.HOME.getValue());
            }
        } else if (null != mappedEmails.email2) {
            vCard.removeProperty(mappedEmails.email2);
        }
        /*
         * email3 - type "X-OTHER", or no specific type
         */
        if (has(contact, Contact.EMAIL3)) {
            if (null == mappedEmails.email3) {
                Email newEmail3 = new Email(contact.getEmail3());
                newEmail3.addParameter(ezvcard.parameter.VCardParameters.TYPE, TYPE_OTHER);
                vCard.addEmail(newEmail3);
            } else {
                mappedEmails.email3.setValue(contact.getEmail3());
                addTypeIfMissing(mappedEmails.email3, TYPE_OTHER);
            }
        } else if (null != mappedEmails.email3) {
            vCard.removeProperty(mappedEmails.email3);
        }
        /*
         * telex - type "TLX"
         */
        if (has(contact, Contact.TELEPHONE_TELEX)) {
            if (null == mappedEmails.telex) {
                vCard.addEmail(contact.getTelephoneTelex(), EmailType.TLX);
            } else {
                mappedEmails.telex.setValue(contact.getTelephoneTelex());
            }
        } else if (null != mappedEmails.telex) {
            vCard.removeProperty(mappedEmails.telex);
        }
    }

    @Override
    public void importVCard(VCard vCard, Contact contact, VCardParameters parameters, List<OXException> warnings) {
        /*
         * skip import for legacy distribution list vCards
         */
        if (isLegacyDistributionList(vCard)) {
            return;
        }
        MappedEmails mappedEmails = extractMappedEmails(vCard);
        contact.setEmail1(parseEMail(mappedEmails.email1, parameters, warnings));
        contact.setEmail2(parseEMail(mappedEmails.email2, parameters, warnings));
        contact.setEmail3(parseEMail(mappedEmails.email3, parameters, warnings));
        contact.setTelephoneTelex(null != mappedEmails.telex ? mappedEmails.telex.getValue() : null);
    }

    /**
     * Extracts all relevant email properties from the given vCard that can be mapped to the available email properties of a contact.
     *
     * @param vCard The vCard to extract the email properties from
     * @return The extracted email properties within a {@link MappedEmails} struct
     */
    private static MappedEmails extractMappedEmails(VCard vCard) {
        List<Email> emails = null == vCard.getEmails() ? Collections.emptyList() : new ArrayList<Email>(vCard.getEmails());
        /*
         * email1 - type "WORK"
         */
        Email email1 = getAndRemoveEmail(vCard, emails, EmailType.WORK.getValue(), null, 0, false);
        /*
         * email2 - type "HOME"
         */
        Email email2 = getAndRemoveEmail(vCard, emails, EmailType.HOME.getValue(), null, 0, false);
        /*
         * email3 - type "X-OTHER", or no specific type
         */
        Email email3 = getAndRemoveEmail(vCard, emails, TYPE_OTHER, ABLABEL_OTHER, 2, true);
        /*
         * telex - type "TLX"
         */
        Email telex = getPropertyWithTypes(emails, EmailType.TLX);
        if (null != telex) {
            emails.remove(telex);
        }
        /*
         * email1, email2, email3 - fill independently of type if candidates left
         */
        if (null == email1 && false == emails.isEmpty()) {
            email1 = emails.remove(0);
        }
        if (null == email2 && false == emails.isEmpty()) {
            email2 = emails.remove(0);
        }
        if (null == email3 && false == emails.isEmpty()) {
            email3 = emails.remove(0);
        }
        return new MappedEmails(email1, email2, email3, telex);
    }

    private static String parseEMail(Email property, VCardParameters parameters, List<OXException> warnings) {
        if (null != property) {
            String value = property.getValue();
            if (Strings.isNotEmpty(value)) {
                if (null == parameters || false == parameters.isValidateContactEMail()) {
                    return value;
                }
                try {
                    new InternetAddress(value).validate();
                    return value;
                } catch (AddressException e) {
                    addConversionWarning(warnings, e, "EMAIL", e.getMessage());
                }
            }
        }
        return null;
    }

    /**
     * Chooses a specific e-mail address from a list of candidates matching either a distinguishing type, or, if the candidates are not
     * using any distinguishing e-mail types at all, the n-th e-mail property as fallback, or, if there are candidates with unknown
     * or no distinguishing types, the first of those e-mail properties as fallback.
     * <p/>
     * If an email property has been chosen successfully, it is removed from the passed list of emails.
     *
     * @param vCard The vCard
     * @param emails The possible e-mail properties to choose from
     * @param distinguishingType The distinguishing type
     * @param abLabel The distinguishing <code>X-ABLabel</code> property, or <code>null</code> if not used
     * @param fallbackIndex The index in the candidate list to use when selecting the fallback property, or <code>-1</code> to use no fallback
     * @param fallbackToUnknownType <code>true</code> to use the first e-mail with unknown distinguishing type as fallback, <code>false</code>, otherwise
     * @return The matching e-mail property, or <code>null</code> if none was found
     */
    private static Email getAndRemoveEmail(VCard vCard, List<Email> emails, String distinguishingType, String abLabel, int fallbackIndex, boolean fallbackToUnknownType) {
        Email email = getEmail(vCard, emails, distinguishingType, abLabel, fallbackIndex, fallbackToUnknownType);
        if (null != email) {
            emails.remove(email);
        }
        return email;
    }

    /**
     * Chooses a specific e-mail address from a list of candidates matching either a distinguishing type, or, if the candidates are not
     * using any distinguishing e-mail types at all, the n-th e-mail property as fallback, or, if there are candidates with unknown
     * or no distinguishing types, the first of those e-mail properties as fallback.
     *
     * @param vCard The vCard
     * @param emails The possible e-mail properties to choose from
     * @param distinguishingType The distinguishing type
     * @param abLabel The distinguishing <code>X-ABLabel</code> property, or <code>null</code> if not used
     * @param fallbackIndex The index in the candidate list to use when selecting the fallback property, or <code>-1</code> to use no fallback
     * @param fallbackToUnknownType <code>true</code> to use the first e-mail with unknown distinguishing type as fallback, <code>false</code>, otherwise
     * @return The matching e-mail property, or <code>null</code> if none was found
     */
    private static Email getEmail(VCard vCard, List<Email> emails, String distinguishingType, String abLabel, int fallbackIndex, boolean fallbackToUnknownType) {
        if (null == emails || 0 == emails.size()) {
            return null;
        }
        /*
         * prefer the most preferred property matching the type
         */
        Email email = getPropertyWithTypes(emails, distinguishingType);
        if (null == email && null != abLabel) {
            /*
             * fallback to an item associated with a matching X-ABLabel
             */
            email = getPropertyWithABLabel(vCard, emails, abLabel);
        }
        if (null == email && 0 <= fallbackIndex) {
            /*
             * if no distinguishing e-mail types defined, use the first address as fallback
             */
            List<Email> simpleEmails = getPropertiesWithoutTypes(emails,
                EmailType.WORK.getValue(), EmailType.HOME.getValue(), TYPE_OTHER, EmailType.TLX.getValue());
            if (fallbackIndex < simpleEmails.size() && simpleEmails.size() == emails.size()) {
                sort(simpleEmails);
                email = simpleEmails.get(fallbackIndex);
            }
        }
        if (null == email && fallbackToUnknownType) {
            /*
             * if no distinguishing e-mail type found, use first non-distinguishing address as fallback, in case there are other distinguishing ones
             */
            List<Email> simpleEmails = getPropertiesWithoutTypes(emails,
                EmailType.WORK.getValue(), EmailType.HOME.getValue(), TYPE_OTHER, EmailType.TLX.getValue());
            if (0 < simpleEmails.size() && simpleEmails.size() != emails.size()) {
                sort(simpleEmails);
                email = simpleEmails.get(0);
            }
        }
        return email;
    }

    /**
     * {@link MappedEmails} - Container for vCard email properties that can be mapped to the available email properties of a contact.
     */
    private static final class MappedEmails {

        final Email email1;
        final Email email2;
        final Email email3;
        final Email telex;

        MappedEmails(Email email1, Email email2, Email email3, Email telex) {
            super();
            this.email1 = email1;
            this.email2 = email2;
            this.email3 = email3;
            this.telex = telex;
        }
    }

}
