/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.config.lean;

/**
 * {@link Properties} - Utility class for lean-configuration properties.
 *
 * @author <a href="mailto:daniel.becker@open-xchange.com">Daniel Becker</a>
 * @since v7.10.1
 */
public final class Properties {

    /**
     * No instantiation.
     */
    private Properties() {
        super();
    }

    /**
     * Get all property names via {@link Property#getFQPropertyName()}.
     *
     * @param values All values of an enum provided by #values()
     * @return An array of {@link String}s.
     */
    public static <T extends Property> String[] getPropertyNames(T[] values) {
        String[] retval = new String[values.length];
        for (int i = 0; i < values.length; i++) {
            retval[i] = values[i].getFQPropertyName();
        }
        return retval;
    }


    // ----------------------------------------------- Property retrieval ------------------------------------------------------------------

    /**
     * Gets the property's <code>String</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The value
     */
    public static String getStringOrDefault(Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(String.class);
        }
        return optConfigurationService.getProperty(property);
    }

    /**
     * Gets the property's <code>long</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The <code>long</code> value
     */
    public static long getLongOrDefault(Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(Long.class).longValue();
        }
        return optConfigurationService.getLongProperty(property);
    }

    /**
     * Gets the property <code>int</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The <code>int</code> value
     */
    public static int getIntOrDefault(Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(Integer.class).intValue();
        }
        return optConfigurationService.getIntProperty(property);
    }

    /**
     * Gets the property's <code>boolean</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The <code>boolean</code> value
     */
    public static boolean getBooleanOrDefault(Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(Boolean.class).booleanValue();
        }
        return optConfigurationService.getBooleanProperty(property);
    }

    /**
     * Gets the property's <code>float</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The <code>float</code> value
     */
    public static float getFloatOrDefault(Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(Float.class).floatValue();
        }
        return optConfigurationService.getFloatProperty(property);
    }

    // ----------------------------------------------- User-specific property retrieval ----------------------------------------------------

    /**
     * Gets the property's <code>String</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The value
     */
    public static String getStringOrDefault(int userId, int contextId, Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(String.class);
        }
        return optConfigurationService.getProperty(userId, contextId, property);
    }

    /**
     * Gets the property's <code>long</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The <code>long</code> value
     */
    public static long getLongOrDefault(int userId, int contextId, Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(Long.class).longValue();
        }
        return optConfigurationService.getLongProperty(userId, contextId, property);
    }

    /**
     * Gets the property <code>int</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The <code>int</code> value
     */
    public static int getIntOrDefault(int userId, int contextId, Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(Integer.class).intValue();
        }
        return optConfigurationService.getIntProperty(userId, contextId, property);
    }

    /**
     * Gets the property's <code>boolean</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The <code>boolean</code> value
     */
    public static boolean getBooleanOrDefault(int userId, int contextId, Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(Boolean.class).booleanValue();
        }
        return optConfigurationService.getBooleanProperty(userId, contextId, property);
    }

    /**
     * Gets the property's <code>float</code> value or its default value if given lean-configuration service is <code>null</code>.
     *
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param property The property
     * @param optConfigurationService The lean-configuration service or <code>null</code>
     * @return The <code>float</code> value
     */
    public static float getFloatOrDefault(int userId, int contextId, Property property, LeanConfigurationService optConfigurationService) {
        if (optConfigurationService == null) {
            return property.getDefaultValue(Float.class).floatValue();
        }
        return optConfigurationService.getFloatProperty(userId, contextId, property);
    }

}
