/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.chronos.scheduling.analyzers;

import static com.openexchange.chronos.common.CalendarUtils.find;
import static com.openexchange.chronos.common.CalendarUtils.getFlags;
import static com.openexchange.chronos.common.CalendarUtils.isInternal;
import static com.openexchange.chronos.common.CalendarUtils.optEMailAddress;
import static com.openexchange.chronos.common.CalendarUtils.optExtendedPropertyValue;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.TimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.chronos.Attendee;
import com.openexchange.chronos.CalendarObjectResource;
import com.openexchange.chronos.CalendarUser;
import com.openexchange.chronos.CalendarUserType;
import com.openexchange.chronos.Event;
import com.openexchange.chronos.EventField;
import com.openexchange.chronos.common.CalendarUtils;
import com.openexchange.chronos.scheduling.AnalyzedChange;
import com.openexchange.chronos.scheduling.ITipAnalysis;
import com.openexchange.chronos.scheduling.ITipChange;
import com.openexchange.chronos.scheduling.ITipChange.Type;
import com.openexchange.chronos.scheduling.SchedulingMethod;
import com.openexchange.chronos.service.CalendarSession;
import com.openexchange.exception.OXException;
import com.openexchange.i18n.LocaleTools;
import com.openexchange.java.Strings;
import com.openexchange.tools.arrays.Collections;

/**
 * {@link Utils}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 * @since v7.10.6
 */
public class Utils {

    private final static Logger LOG = LoggerFactory.getLogger(Utils.class);

    /**
     * Safely gets the summary of the supplied event, falling back to an empty string if there is none.
     * 
     * @param event The event to get the summary from
     * @return The summary, or an empty string if there is none
     */
    public static String getSummary(Event event) {
        return null != event && Strings.isNotEmpty(event.getSummary()) ? event.getSummary() : "";
    }

    /**
     * Safely gets the summary of the supplied event, falling back to an alternative event and finally an empty string if there is none.
     * 
     * @param event The event to get the summary from
     * @param alternativeEvent The alternative event to get the summary from
     * @return The summary, or an empty string if there is none
     */
    public static String getSummary(Event event, Event alternativeEvent) {
        String summary = getSummary(event);
        return Strings.isNotEmpty(summary) ? summary : getSummary(alternativeEvent);
    }

    /**
     * Safely gets the locale of the calendar session user.
     * 
     * @param session The session to derive the locale for
     * @return The locale
     */
    public static Locale getLocale(CalendarSession session) {
        try {
            return session.getEntityResolver().getLocale(session.getUserId());
        } catch (Exception e) {
            LOG.warn("Unexpected error getting locale for session user, falling back to default locale.", e);
            return LocaleTools.DEFAULT_LOCALE;
        }
    }

    /**
     * Safely gets the timezone of the calendar session user.
     * 
     * @param session The session to derive the timezone for
     * @return The timezone
     */
    public static TimeZone getTimeZone(CalendarSession session) {
        try {
            return session.getEntityResolver().getTimeZone(session.getUserId());
        } catch (Exception e) {
            LOG.warn("Unexpected error getting timezone for session user, falling back to default timezone.", e);
            return TimeZone.getDefault();
        }
    }

    /**
     * Constructs a display name for a certain calendar user to be used in explanatory annotations of the analysis.
     * 
     * @param calendarUser The calendar user to get the display name for
     * @return The display name for the calendar user
     */
    public static String getDisplayName(CalendarUser calendarUser) {
        return getDisplayName(calendarUser, null);
    }

    /**
     * Constructs a display name for a certain calendar user to be used in explanatory annotations of the analysis.
     * 
     * @param session The calendar session
     * @param calendarUserId The identifier of the internal calendar user to get the display name for
     * @return The display name for the calendar user
     * @throws OXException In case user can't be found
     */
    public static String getDisplayName(CalendarSession session, int calendarUserId) throws OXException {
        return getDisplayName(session.getEntityResolver().prepareUserAttendee(calendarUserId));
    }

    /**
     * Constructs a display name for a certain calendar user to be used in explanatory annotations of the analysis.
     * 
     * @param calendarUser The calendar user to get the display name for
     * @param originalResource The original calendar object resource, or <code>null</code> if not available
     * @return The display name for the calendar user
     */
    public static String getDisplayName(CalendarUser calendarUser, CalendarObjectResource originalResource) {
        CalendarUser originalUser = find(originalResource, calendarUser);
        String cn = calendarUser.getCn();
        if (Strings.isEmpty(cn) && null != originalUser) {
            cn = originalUser.getCn();
        }
        String email = calendarUser.getEMail(); // try email property first for display purposes
        if (Strings.isEmpty(email) && null != originalUser) {
            email = originalUser.getEMail();
        }
        if (Strings.isEmpty(email)) {
            email = optEMailAddress(calendarUser.getUri());
            if (Strings.isEmpty(email) && null != originalUser) {
                email = optEMailAddress(originalUser.getUri());
            }
        }
        if (Strings.isEmpty(email)) {
            return Strings.isEmpty(cn) ? calendarUser.getUri() : cn;
        }
        if (Strings.isEmpty(cn) || cn.equals(email)) {
            return email;
        }
        return isInternal(calendarUser, CalendarUserType.INDIVIDUAL) ? cn : String.format("%1$s <%2$s>", cn, email);
    }

    /**
     * Creates the resulting iTIP analysis from a list of individual changes for the targeted calendar object resource.
     * 
     * @param method The scheduling method to take over in the analysis
     * @param uid The unique identifier of the scheduling object resource
     * @param analyzedChanges The analyzed changes
     * @param originalResource The currently stored original calendar object resource, or <code>null</code> if there is none
     * @return The resulting iTIP analysis
     */
    protected static ITipAnalysis getAnalysis(SchedulingMethod method, String uid, List<AnalyzedChange> analyzedChanges, CalendarObjectResource originalResource) {
        ITipAnalysis analysis = new ITipAnalysis();
        analysis.setMethod(method);
        analysis.setUid(uid);
        analysis.setOriginalResource(originalResource);
        analysis.setChanges(analyzedChanges);
        analysis.setMainChange(findMainChange(analyzedChanges));
        return analysis;
    }

    /**
     * Prepares an analysis result representing insufficient permissions to process the incoming scheduling message.
     * 
     * @param method The scheduling method to take over in the analysis
     * @param uid The unique identifier of the scheduling object resource
     * @return The resulting iTIP analysis
     */
    protected static ITipAnalysis getInsufficientPermissionsAnalysis(SchedulingMethod method, String uid) {
        ITipAnalysis analysis = new ITipAnalysis();
        analysis.setMethod(method);
        analysis.setUid(uid);
        return analysis;
    }

    /**
     * Optionally gets a comment included in the supplied event's extended properties that may have been set by the scheduling message's
     * originator.
     * 
     * @param event The event to get the comment from
     * @return The optional comment, or an empty optional if there is none
     */
    public static Optional<String> optComment(Event event) {
        String value = optExtendedPropertyValue(event.getExtendedProperties(), "COMMENT", String.class);
        return Strings.isEmpty(value) ? Optional.empty() : Optional.of(value.trim());
    }

    /**
     * Creates a copy of an event from an incoming scheduling object resource and applies certain patches to it to ease further
     * processing. This includes:
     * <ul>
     * <li>adjusting the timezones referenced by the date properties to match a known and supported timezone</li>
     * <li>resolving the current calendar- and session user within the event's attendees and organizer</li>
     * <li>setting the event flags to provide hints for the client</li>
     * </ul>
     * 
     * @param session The underlying calendar session
     * @param event The event from the incoming scheduling object resource to patch
     * @param storedResource The currently stored calendar object resource the message is targeted at, or <code>null</code> if not applicable
     * @param calendarUserId The effective calendar user id
     * @return The patched event
     */
    static Event patchEvent(CalendarSession session, Event event, CalendarObjectResource storedResource, int calendarUserId) {
        Event originalEvent = null != storedResource ? storedResource.getFirstEvent() : null;
        try {
            Event patchedEvent = session.getUtilities().copyEvent(event, (EventField[]) null);
            session.getUtilities().adjustTimeZones(session.getSession(), calendarUserId, patchedEvent, originalEvent);
            session.getEntityResolver().prepare(patchedEvent.getAttendees(), new int[] { calendarUserId, session.getUserId() });
            session.getEntityResolver().prepare(patchedEvent.getOrganizer(), CalendarUserType.INDIVIDUAL);
            patchedEvent.setFlags(getFlags(patchedEvent, calendarUserId, session.getUserId()));
            return patchedEvent;
        } catch (OXException e) {
            session.addWarning(e);
            LOG.warn("Unexpected error patching {}, falling back to original representation", event, e);
            return event;
        }
    }

    /**
     * Selects the originating attendee from an incoming event, based on the scheduling messages originator.
     * 
     * @param incomingEvent The incoming event to select the attendee in
     * @param originator The originator of the scheduling message
     * @return The attendee, or <code>null</code> if no matching attendee is present in the event
     */
    static Attendee selectAttendee(Event incomingEvent, CalendarUser originator) {
        return selectAttendee(incomingEvent, originator, false);
    }

    /**
     * Selects the originating attendee from an incoming event, based on the scheduling messages originator.
     * 
     * @param incomingEvent The incoming event to select the attendee in
     * @param originator The originator of the scheduling message
     * @param mustMatch <code>true</code> if the attende <i>must</i> match the originator, <code>false</code> to allow falling back to the first/only attendee
     * @return The attendee, or <code>null</code> if no matching attendee is present in the event
     */
    static Attendee selectAttendee(Event incomingEvent, CalendarUser originator, boolean mustMatch) {
        List<Attendee> attendees = incomingEvent.getAttendees();
        if (null != attendees && 0 < attendees.size()) {
            /*
             * select attendee based on originator if applicable
             */
            if (null != originator) {
                if (mustMatch) {
                    return find(attendees, originator);
                }
                if (1 < attendees.size()) {
                    Attendee attendee = find(attendees, originator);
                    if (null != attendee) {
                        return attendee;
                    }
                }
            }
            /*
             * use first / only attendee, otherwise
             */
            return attendees.get(0);
        }
        return null;
    }

    /**
     * Gets an attendee representation for the supplied calendar user.
     * 
     * @param calendarUser The calendar user to get the attendee from
     * @param cuType The calendar user type to apply
     * @return The attendee
     */
    static Attendee asAttendee(CalendarUser calendarUser, CalendarUserType cuType) {
        if (null == calendarUser) {
            return null;
        }
        Attendee attendee = new Attendee();
        attendee.setCuType(cuType);
        attendee.setCn(calendarUser.getCn());
        attendee.setEMail(calendarUser.getEMail());
        attendee.setEntity(calendarUser.getEntity());
        attendee.setUri(calendarUser.getUri());
        attendee.setSentBy(calendarUser.getSentBy());
        return attendee;
    }

    /**
     * Finds the change that best fits the main change. Can be either
     * <li> the only change available for non-series event</li>
     * <li> the change about the master</li>
     * <li> any change about an exception, if no master change is present</li>
     * 
     * @param analysis The analysis
     * @return The single main change, or <code>null</code> if the changes are empty
     */
    private static AnalyzedChange findMainChange(List<AnalyzedChange> analyzedChanges) {
        if (Collections.isNullOrEmpty(analyzedChanges)) {
            return null;
        }
        /*
         * No series or single event instance
         */
        if (analyzedChanges.size() == 1) {
            return analyzedChanges.get(0);
        }
        /*
         * Search for a master event
         */
        for (AnalyzedChange change : analyzedChanges) {
            ITipChange iTipChange = change.getChange();
            Event newEvent = iTipChange.getNewEvent();
            if (Type.UPDATE.equals(iTipChange.getType()) || Type.CREATE.equals(iTipChange.getType())) {
                if (CalendarUtils.looksLikeSeriesMaster(newEvent)) {
                    return change;
                }
            }
        }
        /*
         * Fallback to describe only the first change
         */
        return analyzedChanges.get(0);
    }

}
