/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.chronos.scheduling.analyzers;

import static com.openexchange.chronos.common.CalendarUtils.find;
import static com.openexchange.chronos.common.CalendarUtils.isSimilarICloudIMipMeCom;
import static com.openexchange.chronos.common.CalendarUtils.matches;
import static com.openexchange.chronos.common.CalendarUtils.optExtendedParameterValue;
import static com.openexchange.chronos.scheduling.analyzers.Utils.optComment;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.chronos.Attendee;
import com.openexchange.chronos.CalendarUser;
import com.openexchange.chronos.Event;
import com.openexchange.chronos.exception.CalendarExceptionCodes;
import com.openexchange.chronos.scheduling.AnalyzedChange;
import com.openexchange.chronos.scheduling.ITipAction;
import com.openexchange.chronos.scheduling.ITipAnnotation;
import com.openexchange.chronos.scheduling.ITipChange.Type;
import com.openexchange.chronos.scheduling.ITipSequence;
import com.openexchange.chronos.scheduling.SchedulingMethod;
import com.openexchange.chronos.scheduling.analyzers.annotations.AnnotationHelper;
import com.openexchange.chronos.scheduling.changes.Change;
import com.openexchange.chronos.scheduling.common.DefaultChange;
import com.openexchange.chronos.service.CalendarSession;
import com.openexchange.exception.OXException;
import com.openexchange.server.ServiceLookup;

/**
 * {@link RequestAnalyzer} - Analyzer for the iTIP method <code>REQUEST</code>
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 * @since v7.10.6
 * @see <a href="https://datatracker.ietf.org/doc/html/rfc5546#section-3.2.2">RFC 5546 Section 3.2.2</a>
 */
public class RequestAnalyzer extends AbstractSchedulingAnalyzer {

    private final static Logger LOG = LoggerFactory.getLogger(RequestAnalyzer.class);

    /**
     * Initializes a new {@link RequestAnalyzer}.
     * 
     * @param services The services
     */
    public RequestAnalyzer(ServiceLookup services) {
        super(services, SchedulingMethod.REQUEST);
    }

    @Override
    protected List<AnalyzedChange> analyze(CalendarSession session, ObjectResourceProvider objectResourceProvider, CalendarUser originator, int targetUser) throws OXException {
        List<AnalyzedChange> analyzedChanges = new ArrayList<AnalyzedChange>();
        for (Event patchedEvent : objectResourceProvider.getIncomingEvents(true)) {
            Event storedEvent = objectResourceProvider.optMatchingEvent(patchedEvent);
            if (null == storedEvent) {
                Event eventTombstone = objectResourceProvider.optMatchingTombstone(patchedEvent);
                analyzedChanges.add(analyzeUnknownEvent(session, patchedEvent, eventTombstone, originator, targetUser));
            } else {
                analyzedChanges.add(analyzeKnownEvent(session, patchedEvent, storedEvent, originator, targetUser));
            }
        }
        return analyzedChanges;
    }

    /**
     * Analyzes the changes for a single event as part of an incoming {@link SchedulingMethod#REQUEST} from the originator, where no
     * corresponding stored event (occurrence) exists for yet.
     * 
     * @param session The underlying calendar session
     * @param event The (patched) event from the incoming scheduling object resource
     * @param tombstoneEvent The corresponding event tombstone or occurrence, or <code>null</code> if there is none
     * @param originator The originator of the scheduling message
     * @param targetUser The user id of the scheduling message's recipient
     * @return The analyzed change
     */
    private AnalyzedChange analyzeUnknownEvent(CalendarSession session, Event event, Event tombstoneEvent, CalendarUser originator, int targetUser) throws OXException {
        AnnotationHelper annotationHelper = new AnnotationHelper(services, session);
        AnalyzedChange change = new AnalyzedChange();
        if (null != tombstoneEvent && ITipSequence.of(tombstoneEvent).after(ITipSequence.of(event))) {
            /*
             * event has been deleted in the meantime, say so
             */
            change.addAnnotations(getIntroductions(session, event, tombstoneEvent, originator, targetUser));
            change.addAnnotation(annotationHelper.getDeletedHint());
            change.setChange(getChange(session, event, null, Type.CREATE, targetUser, false));
            change.addActions(ITipAction.IGNORE);
        } else {
            /*
             * add introduction about the incoming message and a "save manually" hint, as well as the corresponding actions
             */
            change.addAnnotations(getIntroductions(session, event, null, originator, targetUser));
            change.addAnnotation(annotationHelper.getSaveManuallyHint(targetUser));
            change.addActions(ITipAction.APPLY_CREATE);
            change.setChange(getChange(session, event, null, Type.CREATE, targetUser, true));
            /*
             * add hint about the current participation status and add actions to change it (with or w/o conflicts)
             */
            addPartStatHintAndActions(annotationHelper, change, event, targetUser);
        }
        return change;
    }

    /**
     * Analyzes the changes for a single event as part of an incoming {@link SchedulingMethod#REQUEST} from the originator, where a
     * corresponding stored event (occurrence) exists for.
     * 
     * @param session The underlying calendar session
     * @param event The (patched) event from the incoming scheduling object resource
     * @param storedEvent The corresponding stored event or occurrence
     * @param originator The originator of the scheduling message
     * @param targetUser The user id of the scheduling message's recipient
     * @return The analyzed change
     */
    private AnalyzedChange analyzeKnownEvent(CalendarSession session, Event event, Event storedEvent, CalendarUser originator, int targetUser) throws OXException {
        AnnotationHelper annotationHelper = new AnnotationHelper(services, session);
        AnalyzedChange change = new AnalyzedChange();
        /*
         * add introductional annotation for the incoming message as such
         */
        change.addAnnotations(getIntroductions(session, event, storedEvent, originator, targetUser));
        /*
         * add further annotation(s) based on the current state of the scheduling object resource & derive changes
         */
        if (ITipSequence.of(storedEvent).equals(ITipSequence.of(event))) {
            /*
             * incoming event matches the stored event copy, say so
             */
            if (0 == event.getSequence()) {
                change.addAnnotation(annotationHelper.getSavedHint(targetUser));
                change.setChange(getChange(session, event, storedEvent, Type.CREATE, targetUser, true));
            } else {
                change.addAnnotation(annotationHelper.getUpdatedHint(targetUser));
                change.setChange(getChange(session, event, storedEvent, Type.UPDATE, targetUser, true));
            }
            /*
             * add hint about the current participation status and add actions to change it (with or w/o conflicts)
             */
            addPartStatHintAndActions(annotationHelper, change, storedEvent, targetUser);
        } else if (ITipSequence.of(storedEvent).after(ITipSequence.of(event))) {
            /*
             * event has been updated in the meantime, say so
             */
            change.addAnnotation(annotationHelper.getOutdatedHint());
            change.setChange(getChange(session, event, storedEvent, Type.UPDATE, targetUser, false));
            change.addActions(ITipAction.IGNORE);
        } else if (ITipSequence.of(storedEvent).before(ITipSequence.of(event))) {
            /*
             * technically applicable, but not yet applied, perform further checks & include suggestions as needed
             */
            if (false == matches(storedEvent.getOrganizer(), event.getOrganizer()) && 
                false == isSimilarICloudIMipMeCom(storedEvent.getOrganizer(), event.getOrganizer())) {
                /*
                 * organizer change, add corresponding hint & offer "ignore" action
                 */
                change.addAnnotation(annotationHelper.getOrganizerChangedHint());
                change.addActions(ITipAction.IGNORE);
            }
            change.addAnnotation(annotationHelper.getUpdateManuallyHint(targetUser));
            change.setChange(getChange(session, event, storedEvent, Type.UPDATE, targetUser, true));
            change.addActions(ITipAction.APPLY_CHANGE);
            /*
             * add hint about the current participation status and add actions to change it (with or w/o conflicts)
             */
            addPartStatHintAndActions(annotationHelper, change, event, targetUser);
        } else {
            throw CalendarExceptionCodes.UNEXPECTED_ERROR.create("Illegal sequence/dtstamp in events");
        }
        return change;
    }

    /**
     * Gets the introductional annotation(s) describing the event included in an incoming {@link SchedulingMethod#REQUEST} from the
     * originator.
     * 
     * @param session The calendar session
     * @param event The (patched) event or occurrence from the incoming scheduling object resource
     * @param storedEvent The corresponding currently stored event or occurrence, or <code>null</code> if not applicable
     * @param originator The originator of the scheduling message
     * @param targetUser The user id of the scheduling message's recipient
     * @return The introductional annotations
     */
    private List<ITipAnnotation> getIntroductions(CalendarSession session, Event event, Event storedEvent, CalendarUser originator, int targetUser) throws OXException {
        List<ITipAnnotation> annotations = new ArrayList<ITipAnnotation>(2);
        AnnotationHelper annotationHelper = new AnnotationHelper(services, session);
        if (null == storedEvent || 0 == event.getSequence()) {
            Attendee delegator = optDelegator(event, targetUser);
            if (null != delegator) {
                /*
                 * new delegated appointment, use "delegated" annotations
                 */
                annotations.add(annotationHelper.getDelegatedIntroduction(event, originator, delegator, targetUser));
            } else {
                /*
                 * new appointment (from targeted user's point of view), use "invited to" annotations
                 */
                annotations.add(annotationHelper.getInvitedIntroduction(event, originator, targetUser));
            }
        } else {
            /*
             * update of existing appointment (from targeted user's point of view), use "has updated" annotations
             */
            annotations.add(annotationHelper.getChangedIntroduction(event, originator));
        }
        optComment(event).ifPresent((c) -> annotations.add(annotationHelper.getRequestCommentHint(c)));
        return annotations;
    }

    private static Attendee optDelegator(Event event, int targetUser) {
        Attendee calendarUserAttendee = find(event.getAttendees(), targetUser);
        if (null != calendarUserAttendee) {
            String delegatedFrom = optExtendedParameterValue(calendarUserAttendee.getExtendedParameters(), "DELEGATED-FROM");
            return find(event.getAttendees(), delegatedFrom);
        }
        return null;
    }

    /**
     * Constructs a minimal {@link Change} for an event from a {@link SchedulingMethod#REQUEST} message.
     * 
     * @param session The calendar session
     * @param incomingEvent The (patched) event from the incoming message to get the change for
     * @param storedEvent The corresponding currently stored event or occurrence, or <code>null</code> if not applicable
     * @param type The type of change, either {@link Type#UPDATE} or {@link Type#CREATE}
     * @param targetUser The user id of the scheduling message's recipient
     * @param checkConflicts <code>true</code> to check for conflicts in the targeted user's calendar, <code>false</code> otherwise
     * @return The change
     */
    private static DefaultChange getChange(CalendarSession session, Event incomingEvent, Event storedEvent, Type type, int targetUser, boolean checkConflicts) {
        DefaultChange change = new DefaultChange();
        change.setType(type);
        change.setNewEvent(incomingEvent);
        change.setCurrentEvent(storedEvent);
        if (checkConflicts) {
            try {
                Attendee attendee = session.getEntityResolver().prepareUserAttendee(targetUser);
                change.setConflicts(session.getFreeBusyService().checkForConflicts(session, incomingEvent, Collections.singletonList(attendee)));
            } catch (OXException e) {
                LOG.warn("Unexpected error checking for conflicts for incoming event from REQUEST message", e);
                session.addWarning(e);
            }
        }
        return change;
    }

    /**
     * Adds a hint about the current participation status and actions to change it (with or w/o conflicts) to the passed analyzed change.
     * 
     * @param annotationHelper The annotation helper to use
     * @param change The analyzed change to populate with the actions and annotation
     * @param event The incoming event in question
     * @param targetUser The user id of the scheduling message's recipient
     */
    private static void addPartStatHintAndActions(AnnotationHelper annotationHelper, AnalyzedChange change, Event event, int targetUser) throws OXException {
        change.addAnnotation(annotationHelper.getPartStatDescription(event, targetUser));
        if (com.openexchange.tools.arrays.Collections.isNotEmpty(change.getChange().getConflicts())) {
            change.addAnnotation(annotationHelper.getConflictsHint(targetUser));
            change.addActions(ITipAction.DECLINE, ITipAction.TENTATIVE, ITipAction.ACCEPT_AND_IGNORE_CONFLICTS);
        } else {
            change.addActions(ITipAction.DECLINE, ITipAction.TENTATIVE, ITipAction.ACCEPT);
        }
    }

}
