/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.chronos.scheduling.analyzers;

import static com.openexchange.chronos.common.CalendarUtils.getOccurrence;
import static com.openexchange.chronos.scheduling.analyzers.Utils.patchEvent;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.chronos.CalendarObjectResource;
import com.openexchange.chronos.Event;
import com.openexchange.chronos.EventField;
import com.openexchange.chronos.RecurrenceId;
import com.openexchange.chronos.common.DefaultCalendarObjectResource;
import com.openexchange.chronos.scheduling.IncomingSchedulingMessage;
import com.openexchange.chronos.service.CalendarParameters;
import com.openexchange.chronos.service.CalendarSession;
import com.openexchange.exception.OXException;

/**
 * {@link ObjectResourceProvider}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 * @since v7.10.6
 */
public class ObjectResourceProvider {

    /** The default event fields to load when retrieving the currently stored calendar object resources from the storage */
    private static final EventField[] DEFAULT_FIELDS = { //@formatter:off
        EventField.ID, EventField.SERIES_ID, EventField.FOLDER_ID, EventField.UID, EventField.RECURRENCE_ID, EventField.RECURRENCE_RULE,
        EventField.RECURRENCE_DATES, EventField.DELETE_EXCEPTION_DATES, EventField.CHANGE_EXCEPTION_DATES, EventField.START_DATE, 
        EventField.SEQUENCE, EventField.DTSTAMP, EventField.ORGANIZER, EventField.ATTENDEES, EventField.SUMMARY
    }; //@formatter:on

    private final static Logger LOG = LoggerFactory.getLogger(ObjectResourceProvider.class);

    private final CalendarSession session;
    private final String uid;
    private final int calendarUserId;
    private final EventField[] fieldsToLoad;
    private final IncomingSchedulingMessage incomingMessage;

    private Optional<CalendarObjectResource> storedResource;
    private Optional<CalendarObjectResource> tombstoneResource;

    /**
     * Initializes a new {@link ObjectResourceProvider}.
     * <p/>
     * The currently stored calendar object resources are loaded with the {@link #DEFAULT_FIELDS} from the storage.
     * 
     * @param session The calendar session
     * @param incomingMessage The incoming scheduling message
     */
    public ObjectResourceProvider(CalendarSession session, IncomingSchedulingMessage incomingMessage) {
        this(session, incomingMessage, DEFAULT_FIELDS);
    }

    /**
     * Initializes a new {@link ObjectResourceProvider}.
     * 
     * @param session The calendar session
     * @param incomingMessage The incoming scheduling message
     * @param fieldsToLoad The event fields to load when retrieving the currently stored calendar object resources from the storage, or <code>null</code> to load all fields
     */
    public ObjectResourceProvider(CalendarSession session, IncomingSchedulingMessage incomingMessage, EventField[] fieldsToLoad) {
        super();
        this.session = session;
        this.incomingMessage = incomingMessage;
        this.uid = incomingMessage.getResource().getUid();
        this.calendarUserId = incomingMessage.getTargetUser();
        this.fieldsToLoad = fieldsToLoad;
    }

    /**
     * Get additional information.
     * 
     * @param key The key for the value
     * @param clazz The class the value has
     * @return An Optional holding the value casted to the given class
     * @param <T> The class of the returned object
     */
    public <T> Optional<T> getAdditional(String key, Class<T> clazz) {
        return incomingMessage.getAdditional(key, clazz);
    }

    /**
     * Gets the calendar object resource from the incoming scheduling message.
     * 
     * @return The incoming calendar object resource
     */
    public CalendarObjectResource getIncomingResource() {
        return incomingMessage.getResource();
    }

    /**
     * Gets the individual events from the incoming calendar object resource.
     * <p/>
     * Optionally, the generic patch routine from {@link Utils#patchEvent} is applied.
     * 
     * @param patch <code>true</code> to apply patched for each event, <code>false</code>, otherwise
     * @return The the individual events from the incoming calendar object resource
     */
    public List<Event> getIncomingEvents(boolean patch) {
        List<Event> events = getIncomingResource().getEvents();
        if (null == events || events.isEmpty() || false == patch) {
            return events;
        }
        CalendarObjectResource resourceForPatching = optResourceForPatching();
        return events.stream().map((e) -> patchEvent(session, e, resourceForPatching, calendarUserId)).collect(Collectors.toList());
    }

    /**
     * Resolves an UID to all stored events belonging to the corresponding calendar object resource. The lookup is performed case-
     * sensitive, within the scope of a specific calendar user. I.e., the unique identifier is resolved to events residing in the user's
     * <i>personal</i>, as well as <i>public</i> calendar folders.
     * <p/>
     * The events will be <i>userized</i> to reflect the view of the calendar user on the events.
     * 
     * @return The <i>userized</i> events as calendar object resource, or <code>null</code> if no events were found
     * @throws OXException If loading of the events fails
     */
    public CalendarObjectResource getStoredResource() throws OXException {
        if (null == storedResource) {
            List<Event> events = getStoredEvents(false);
            storedResource = events.isEmpty() ? Optional.empty() : Optional.of(new DefaultCalendarObjectResource(events));
        }
        return storedResource.orElse(null);
    }

    /**
     * Resolves an UID to all events belonging to the corresponding calendar object resource, as found in the <i>tombstone</i> storage.
     * The lookup is performed case-sensitive, within the scope of a specific calendar user. I.e., the unique identifier is resolved to
     * events that were previously residing in the user's <i>personal</i>, as well as <i>public</i> calendar folders.
     * <p/>
     * The events will be <i>userized</i> to reflect the view of the calendar user on the events.
     * 
     * @return The <i>userized</i> event tombstones as calendar object resource, or <code>null</code> if no events were found
     * @throws OXException If loading of the events fails
     */
    public CalendarObjectResource getTombstoneResource() throws OXException {
        if (null == tombstoneResource) {
            List<Event> events = getStoredEvents(true);
            tombstoneResource = events.isEmpty() ? Optional.empty() : Optional.of(new DefaultCalendarObjectResource(events));
        }
        return tombstoneResource.orElse(null);
    }

    /**
     * Optionally gets the <i>matching</i> event or occurrence from the currently stored calendar object resource for a specific event
     * from an incoming scheduling message.
     * 
     * @param incomingEvent The incoming event to get the corresponding event for
     * @return The corresponding event (or virtual event occurrence), or <code>null</code> if none could be derived
     * @throws OXException If loading of the events fails
     */
    public Event optMatchingEvent(Event incomingEvent) throws OXException {
        return optMatchingEvent(session, incomingEvent, getStoredResource());
    }

    /**
     * Optionally gets the <i>matching</i> tombstone event or occurrence from the currently stored calendar object resource for a specific
     * event from an incoming scheduling message.
     * 
     * @param incomingEvent The incoming event to get the corresponding tombstone event for
     * @return The corresponding tombstone event (or virtual event occurrence), or <code>null</code> if none could be derived
     * @throws OXException If loading of the events fails
     */
    public Event optMatchingTombstone(Event incomingEvent) throws OXException {
        return optMatchingEvent(session, incomingEvent, getTombstoneResource());
    }

    /**
     * Optionally gets the <i>matching</i> event or occurrence from the given calendar object resource for a specific event from an
     * incoming scheduling message.
     * 
     * @param incomingEvent The incoming event to get the corresponding event for
     * @param resource The calendar object resource to get the matching event from, or <code>null</code> if there is none
     * @return The corresponding event (or virtual event occurrence), or <code>null</code> if none could be derived
     */
    private static Event optMatchingEvent(CalendarSession session, Event incomingEvent, CalendarObjectResource resource) {
        if (null == resource) {
            return null;
        }
        if (null != incomingEvent.getRecurrenceId()) {
            /*
             * match existing change exception or event occurrence
             */
            Event originalChangeException = resource.getChangeException(incomingEvent.getRecurrenceId());
            if (null != originalChangeException) {
                return originalChangeException;
            }
            if (null != resource.getSeriesMaster()) {
                return optEventOccurrence(session, resource.getSeriesMaster(), incomingEvent.getRecurrenceId());
            }
            return null;
        }
        if (null != resource.getFirstEvent() && null == resource.getFirstEvent().getRecurrenceId()) {
            /*
             * match series master or non-recurring
             */
            return resource.getFirstEvent();
        }
        return null;
    }

    private static Event optEventOccurrence(CalendarSession session, Event seriesMaster, RecurrenceId recurrenceId) {
        if (null == seriesMaster) {
            return null;
        }
        try {
            return getOccurrence(session.getRecurrenceService(), seriesMaster, recurrenceId);
        } catch (OXException e) {
            session.addWarning(e);
            LOG.warn("Unexpected error preparing event occurrence for {} of {}", recurrenceId, seriesMaster, e);
            return null;
        }
    }
    
    private List<Event> getStoredEvents(boolean tombstones) throws OXException {
        EventField[] oldParameterFields = session.get(CalendarParameters.PARAMETER_FIELDS, EventField[].class);
        try {
            session.set(CalendarParameters.PARAMETER_FIELDS, fieldsToLoad);
            if (tombstones) {
                return session.getCalendarService().getUtilities().resolveDeletedEventsByUID(session, uid, calendarUserId);
            }
            return session.getCalendarService().getUtilities().resolveEventsByUID(session, uid, calendarUserId);
        } finally {
            session.set(CalendarParameters.PARAMETER_FIELDS, oldParameterFields);
        }
    }

    private CalendarObjectResource optResourceForPatching() {
        CalendarObjectResource resourceForPatching = null;
        try {
            resourceForPatching = getStoredResource();
        } catch (OXException e) {
            LOG.debug("Unable to get stored resource for patching", e);
        }
        if (null == resourceForPatching) {
            try {
                resourceForPatching = getTombstoneResource();
            } catch (OXException e) {
                LOG.debug("Unable to get tombstone resource for patching", e);
            }
        }
        return resourceForPatching;
    }

}
