/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.chronos.impl.performer;

import static com.openexchange.chronos.common.CalendarUtils.asExternalEvents;
import static com.openexchange.chronos.common.CalendarUtils.looksLikeSeriesMaster;
import static com.openexchange.chronos.common.FreeBusyUtils.mergeFreeBusy;
import static com.openexchange.chronos.impl.Utils.anonymizeIfNeeded;
import static com.openexchange.chronos.impl.performer.FreeBusyPerformerUtil.RESTRICTED_FREEBUSY_FIELDS;
import static com.openexchange.chronos.impl.performer.FreeBusyPerformerUtil.adjustToBoundaries;
import static com.openexchange.chronos.impl.performer.FreeBusyPerformerUtil.getFreeBusyResults;
import static com.openexchange.chronos.impl.performer.FreeBusyPerformerUtil.getFreeBusyTime;
import static com.openexchange.chronos.impl.performer.FreeBusyPerformerUtil.includeForFreeBusy;
import static com.openexchange.chronos.impl.performer.FreeBusyPerformerUtil.isVisible;
import static com.openexchange.chronos.impl.performer.FreeBusyPerformerUtil.mergeFreeBusy;
import static com.openexchange.chronos.impl.performer.FreeBusyPerformerUtil.resolveAttendees;
import static com.openexchange.chronos.impl.performer.FreeBusyPerformerUtil.separateAttendees;
import static com.openexchange.chronos.service.CalendarParameters.PARAMETER_MASK_UID;
import static com.openexchange.java.Autoboxing.i;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TimeZone;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.chronos.Attendee;
import com.openexchange.chronos.Event;
import com.openexchange.chronos.FreeBusyTime;
import com.openexchange.chronos.RecurrenceId;
import com.openexchange.chronos.common.DefaultRecurrenceData;
import com.openexchange.chronos.common.EventOccurrence;
import com.openexchange.chronos.common.mapping.EventMapper;
import com.openexchange.chronos.exception.CalendarExceptionCodes;
import com.openexchange.chronos.impl.OccurrenceId;
import com.openexchange.chronos.impl.osgi.Services;
import com.openexchange.chronos.impl.session.CalendarConfigImpl;
import com.openexchange.chronos.service.CalendarSession;
import com.openexchange.chronos.service.FreeBusyResult;
import com.openexchange.chronos.storage.CalendarStorage;
import com.openexchange.exception.OXException;
import com.openexchange.server.ServiceLookup;
import com.openexchange.threadpool.ThreadPools;

/**
 * {@link FreeBusyPerformer}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 * @since v7.10.0
 */
public class FreeBusyPerformer extends AbstractFreeBusyPerformer {

    private static final Logger LOG = LoggerFactory.getLogger(FreeBusyPerformer.class);

    /**
     * Initializes a new {@link FreeBusyPerformer}.
     *
     * @param storage The underlying calendar storage
     * @param session The calendar session
     */
    public FreeBusyPerformer(CalendarSession session, CalendarStorage storage) {
        super(session, storage);
    }

    /**
     * Performs the free/busy operation.
     *
     * @param attendees The attendees to get the free/busy data for
     * @param from The start of the requested time range
     * @param until The end of the requested time range
     * @param merge <code>true</code> to merge the resulting free/busy-times, <code>false</code>, otherwise
     * @return The free/busy times for each of the requested attendees, wrapped within a free/busy result structure
     */
    public Map<Attendee, FreeBusyResult> perform(List<Attendee> attendees, Date from, Date until, boolean merge) throws OXException {
        if (null == attendees || attendees.isEmpty()) {
            return Collections.emptyMap();
        }
        /*
         * resolve passed attendees prior lookup & distinguish between internal, and external user attendees as cross-context candidates
         */
        Map<Attendee, Attendee> resolvedAttendees = resolveAttendees(session.getEntityResolver(), attendees);
        List<Attendee> internalAttendees = new ArrayList<Attendee>(resolvedAttendees.size());
        Map<String, Attendee> externalUserAttendeesByEmail = new HashMap<String, Attendee>();
        separateAttendees(resolvedAttendees.keySet(), internalAttendees, externalUserAttendeesByEmail);
        /*
         * check and initiate cross-context lookup if applicable
         */
        Future<Map<Attendee, List<Event>>> crossContextLookupFuture = null;
        if (0 < externalUserAttendeesByEmail.size() && session.getConfig().isCrossContextFreeBusy()) {
            String maskUid = session.get(PARAMETER_MASK_UID, String.class, null);
            crossContextLookupFuture = ThreadPools.submitElseExecute(new CrossContextFreeBusyTask(Services.getServiceLookup(), externalUserAttendeesByEmail, from, until, maskUid));
        }
        /*
         * get intersecting events per resolved internal attendee and insert into overall result
         */
        Map<Attendee, List<Event>> eventsPerAttendee = new LinkedHashMap<Attendee, List<Event>>(resolvedAttendees.size());
        eventsPerAttendee.putAll(getOverlappingEvents(internalAttendees, from, until));
        /*
         * if available, take over results from users in other contexts, too
         */
        if (null != crossContextLookupFuture) {
            try {
                eventsPerAttendee.putAll(crossContextLookupFuture.get());
            } catch (InterruptedException | ExecutionException e) {
                session.addWarning(CalendarExceptionCodes.UNEXPECTED_ERROR.create(e.getMessage(), e));
            }
        }
        /*
         * derive (merged) free/busy times for found events, mapped back to the client-requested attendees
         */
        Map<OccurrenceId, Event> knownEvents = new HashMap<OccurrenceId, Event>();
        Map<Attendee, List<FreeBusyTime>> freeBusyPerAttendee = new HashMap<Attendee, List<FreeBusyTime>>(eventsPerAttendee.size());
        for (Map.Entry<Attendee, List<Event>> entry : eventsPerAttendee.entrySet()) {
            Attendee attendee = resolvedAttendees.get(entry.getKey());
            if (null == attendee) {
                session.addWarning(CalendarExceptionCodes.UNEXPECTED_ERROR.create("Skipping free/busy times from unexpected attendee " + entry.getKey()));
                continue;
            }
            List<Event> eventInPeriods = entry.getValue();
            if (null == eventInPeriods || eventInPeriods.isEmpty()) {
                freeBusyPerAttendee.put(attendee, Collections.emptyList());
                continue;
            }
            /*
             * create free/busy times for attendee, using anonymized or enriched event data as applicable
             */
            TimeZone timeZone = getTimeZone(attendee);
            List<FreeBusyTime> freeBusyTimes = new ArrayList<FreeBusyTime>(eventInPeriods.size());
            for (Event event : eventInPeriods) {
                if (looksLikeSeriesMaster(event)) {
                    /*
                     * expand & add all (non overridden) instances of event series in period, expanded by the actual event duration
                     */
                    long duration = event.getEndDate().getTimestamp() - event.getStartDate().getTimestamp();
                    Date iterateFrom = new Date(from.getTime() - duration);
                    Iterator<RecurrenceId> iterator = session.getRecurrenceService().iterateRecurrenceIds(new DefaultRecurrenceData(event), iterateFrom, until);
                    while (iterator.hasNext()) {
                        calculateFreeBusyTime(new EventOccurrence(event, iterator.next()), from, until, knownEvents, timeZone, freeBusyTimes);
                    }
                } else {
                    calculateFreeBusyTime(event, from, until, knownEvents, timeZone, freeBusyTimes);
                }
            }
            freeBusyPerAttendee.put(attendee, merge && 1 < freeBusyTimes.size() ? mergeFreeBusy(freeBusyTimes) : freeBusyTimes);
        }
        return getFreeBusyResults(attendees, freeBusyPerAttendee);
    }

    private void calculateFreeBusyTime(Event event, Date from, Date until, Map<OccurrenceId, Event> knownEvents, TimeZone timeZone, List<FreeBusyTime> freeBusyTimes) {
        FreeBusyTime freeBusyTime = getFreeBusyTime(event, timeZone, (e) -> getResultingEvent(e));
        if (null != adjustToBoundaries(freeBusyTime, from, until)) {
            freeBusyTimes.add(rememberInternalOrReplaceExternalEvent(freeBusyTime, knownEvents));
        }
    }

    /**
     * Remembers an <i>internal</i> event (with set object identifiers) from a free/busy timeslot in the supplied map, or re-uses such a
     * previously remembered event for the free/busy timeslot of an <i>external</i> context. Association is done based on the uid /
     * recurrence id pair from {@link OccurrenceId}.
     * 
     * @param freeBusyTime The free/busy time to remember or replace the event in
     * @param knownEvents The previously remembered events, mapped by their {@link OccurrenceId}.
     * @return The (possibly adjusted) free/busy time
     */
    private static FreeBusyTime rememberInternalOrReplaceExternalEvent(FreeBusyTime freeBusyTime, Map<OccurrenceId, Event> knownEvents) {
        Event event = freeBusyTime.getEvent();
        if (null != event) {
            if (null != event.getId()) {
                /*
                 * remember internal event
                 */
                knownEvents.putIfAbsent(new OccurrenceId(event), event);
            } else {
                /*
                 * replace external event
                 */
                Event knownEvent = knownEvents.get(new OccurrenceId(event));
                if (null != knownEvent) {
                    freeBusyTime.setEvent(knownEvent);
                }
            }
        }
        return freeBusyTime;
    }

    /**
     * Adjusts an overlapping event to be used in free/busy results by either preserving the {@link #FREEBUSY_FIELDS} or the
     * {@link #RESTRICTED_FREEBUSY_FIELDS} in the returned event copy, based on the session user's access permission for the event.
     * Optionally, the most appropriate parent folder id representing the session user's view on the event is injected, too.
     * 
     * @param event The event to adjust
     * @return The resulting event, or <code>null</code> if the event cannot be adjusted
     */
    private Event getResultingEvent(Event event) {
        String folderId;
        try {
            folderId = chooseFolderID(event);
        } catch (OXException e) {
            LOG.warn("Unexpected error choosing folder id for event {}", event, e);
            folderId = null;
        }
        try {
            if (null == folderId) {
                return EventMapper.getInstance().copy(event, new Event(), RESTRICTED_FREEBUSY_FIELDS);
            }
            Event resultingEvent = EventMapper.getInstance().copy(event, new Event(), FreeBusyPerformerUtil.FREEBUSY_FIELDS);
            resultingEvent.setFolderId(folderId);
            return anonymizeIfNeeded(session, resultingEvent);
        } catch (OXException e) {
            LOG.warn("Unexpected error adjusting event data {}", event, e);
        }
        return null;
    }

    /**
     * Gets a list of overlapping events in a certain range for each requested attendee.
     *
     * @param attendees The attendees to query free/busy information for
     * @param from The start date of the period to consider
     * @param until The end date of the period to consider
     * @return The overlapping events, mapped to each attendee
     */
    private Map<Attendee, List<Event>> getOverlappingEvents(Collection<Attendee> attendees, Date from, Date until) throws OXException {
        return new OverlappingEventsLoader(storage).load(attendees, from, until, (e, a) -> considerForFreeBusy(e) && includeForFreeBusy(e, a.getEntity()));
    }

    /**
     * Performs the merged free/busy operation.
     *
     * @param attendees The attendees to query free/busy information for
     * @param from The start date of the period to consider
     * @param until The end date of the period to consider
     * @return The free/busy result
     */
    public Map<Attendee, List<FreeBusyTime>> performMerged(List<Attendee> attendees, Date from, Date until) throws OXException {
        Map<Attendee, List<Event>> eventsPerAttendee = getOverlappingEvents(attendees, from, until);
        Map<Attendee, List<FreeBusyTime>> freeBusyDataPerAttendee = new HashMap<Attendee, List<FreeBusyTime>>(eventsPerAttendee.size());
        for (Map.Entry<Attendee, List<Event>> entry : eventsPerAttendee.entrySet()) {
            freeBusyDataPerAttendee.put(entry.getKey(), mergeFreeBusy(entry.getValue(), from, until, getTimeZone(entry.getKey()), (e) -> getResultingEvent(e)));
        }
        return freeBusyDataPerAttendee;
    }

    /**
     * Calculates the free/busy time ranges from the user defined availability and the free/busy operation
     *
     * @param attendees The attendees to calculate the free/busy information for
     * @param from The start time of the interval
     * @param until The end time of the interval
     * @return A {@link Map} with a {@link FreeBusyResult} per {@link Attendee}
     */
    public Map<Attendee, FreeBusyResult> performCalculateFreeBusyTime(List<Attendee> attendees, Date from, Date until) throws OXException {
        // Get the free busy data for the attendees
        Map<Attendee, List<FreeBusyTime>> freeBusyPerAttendee = performMerged(attendees, from, until);
        Map<Attendee, FreeBusyResult> results = new HashMap<>();
        for (Map.Entry<Attendee, List<FreeBusyTime>> attendeeEntry : freeBusyPerAttendee.entrySet()) {
            FreeBusyResult result = new FreeBusyResult();
            result.setFreeBusyTimes(attendeeEntry.getValue());
            results.put(attendeeEntry.getKey(), result);
        }
        return results;
    }

    private static final class CrossContextFreeBusyTask extends AbstractCrossContextLookupTask<Map<Attendee, List<Event>>> {

        private final String maskUid;

        CrossContextFreeBusyTask(ServiceLookup services, Map<String, Attendee> attendeesByMailAdress, Date from, Date until, String maskUid) {
            super(services, attendeesByMailAdress, from, until);
            this.maskUid = maskUid;
        }

        @Override
        public Map<Attendee, List<Event>> call() throws Exception {
            /*
             * resolve attendee's mail addresses to user attendees per context
             */
            Map<Integer, Map<Attendee, Attendee>> attendeesPerContext = resolveToContexts();
            if (null == attendeesPerContext || attendeesPerContext.isEmpty()) {
                return Collections.emptyMap();
            }
            /*
             * get overlapping events for resolved attendees in each context & re-map to incoming attendees
             */
            Map<Attendee, List<Event>> eventsPerAttendee = new HashMap<Attendee, List<Event>>(attendeesByMailAdress.size());
            for (Entry<Integer, Map<Attendee, Attendee>> entry : attendeesPerContext.entrySet()) {
                int contextId = i(entry.getKey());
                if (false == new CalendarConfigImpl(contextId, services).isCrossContextFreeBusy()) {
                    continue;
                }
                Map<Attendee, Attendee> resolvedAttendees = entry.getValue();
                Map<Attendee, List<Event>> overlappingEvents = getOverlappingEventsPerAttendee(
                    contextId, resolvedAttendees.keySet(), (e, a) -> isVisible(e, maskUid) && includeForFreeBusy(e, a.getEntity()));
                for (Entry<Attendee, List<Event>> eventsForAttendee : overlappingEvents.entrySet()) {
                    eventsPerAttendee.put(resolvedAttendees.get(eventsForAttendee.getKey()), asExternalEvents(eventsForAttendee.getValue()));
                }
            }
            return eventsPerAttendee;
        }
    }

}
