/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.chronos.impl.performer;

import static com.openexchange.chronos.common.CalendarUtils.find;
import static com.openexchange.chronos.common.CalendarUtils.isAttendee;
import static com.openexchange.chronos.common.CalendarUtils.isInternal;
import static com.openexchange.chronos.common.CalendarUtils.isOrganizer;
import static com.openexchange.chronos.common.CalendarUtils.isPublicClassification;
import static com.openexchange.chronos.common.CalendarUtils.matches;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.TimeZone;
import com.openexchange.chronos.Attendee;
import com.openexchange.chronos.AttendeeField;
import com.openexchange.chronos.CalendarUserType;
import com.openexchange.chronos.Classification;
import com.openexchange.chronos.Event;
import com.openexchange.chronos.impl.CalendarFolder;
import com.openexchange.chronos.impl.Utils;
import com.openexchange.chronos.service.CalendarParameters;
import com.openexchange.chronos.service.CalendarSession;
import com.openexchange.chronos.storage.CalendarStorage;
import com.openexchange.exception.OXException;
import com.openexchange.folderstorage.Permission;
import com.openexchange.folderstorage.type.PublicType;

/**
 * {@link AbstractFreeBusyPerformer}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 * @since v7.10.0
 */
public class AbstractFreeBusyPerformer extends AbstractQueryPerformer {

    private List<CalendarFolder> visibleFolders;

    /**
     * Initializes a new {@link AbstractFreeBusyPerformer}.
     *
     * @param storage The underlying calendar storage
     * @param session The calendar session
     */
    protected AbstractFreeBusyPerformer(CalendarSession session, CalendarStorage storage) {
        super(session, storage);
    }

    /**
     * Reads the attendee data from storage
     *
     * @param events The events to load
     * @param internal whether to only consider internal attendees or not
     * @return The {@link Event}s containing the attendee data
     * @throws OXException
     */
    protected List<Event> readAttendeeData(List<Event> events, Boolean internal) throws OXException {
        return FreeBusyPerformerUtil.readAttendeeData(events, internal, storage);
    }

    /**
     * Gets the timezone to consider for <i>floating</i> dates of a specific attendee.
     * <p/>
     * For <i>internal</i>, individual calendar user attendees, this is the configured timezone of the user; otherwise, the timezone of
     * the current session's user is used.
     *
     * @param attendee The attendee to get the timezone to consider for <i>floating</i> dates for
     * @return The timezone
     */
    protected TimeZone getTimeZone(Attendee attendee) throws OXException {
        if (isInternal(attendee) && CalendarUserType.INDIVIDUAL.equals(attendee.getCuType())) {
            return session.getEntityResolver().getTimeZone(attendee.getEntity());
        }
        return Utils.getTimeZone(session);
    }

    /**
     * Gets a value indicating whether a certain event is visible or <i>opaque to</i> free/busy results in the view of the current
     * session's user or not.
     *
     * @param event The event to check
     * @return <code>true</code> if the event should be considered, <code>false</code>, otherwise
     */
    protected boolean considerForFreeBusy(Event event) {
        String maskUid = session.get(CalendarParameters.PARAMETER_MASK_UID, String.class);
        if (null != maskUid && maskUid.equals(event.getUid())) {
            return false;
        }

        // exclude foreign events classified as 'private' (but keep 'confidential' ones)
        int userId = session.getUserId();
        return isPublicClassification(event) || Classification.CONFIDENTIAL.equals(event.getClassification()) ||
            matches(event.getCalendarUser(), userId) || isOrganizer(event, userId) || isAttendee(event, userId);
    }

    /**
     * Gets all calendar folders accessible by the current sesssion's user.
     * <p>
     * Folders are always sorted
     *
     * @return The folders, or an empty list if there are none
     */
    protected List<CalendarFolder> getVisibleFolders() throws OXException {
        if (null == visibleFolders) {
            visibleFolders = Utils.getVisibleFolders(session);
        }
        return visibleFolders;
    }

    /**
     * Chooses the most appropriate parent folder identifier to render an event in for the current session's user. This is
     * <ul>
     * <li>the common parent folder identifier for an event in a public folder, in case the user has appropriate folder permissions</li>
     * <li><code>-1</code> for an event in a public folder, in case the user has no appropriate folder permissions</li>
     * <li>the user attendee's personal folder identifier for an event in a non-public folder, in case the user is attendee of the event</li>
     * <li>another attendee's personal folder identifier for an event in a non-public folder, in case the user does not attend on his own, but has appropriate folder permissions for this attendee's folder</li>
     * <li><code>-1</code> for an event in a non-public folder, in case the user has no appropriate folder permissions for any of the attendees</li>
     * </ul>
     *
     * @param event The event to choose the folder identifier for
     * @return The chosen folder identifier, or <code>null</code> if there is none
     */
    protected String chooseFolderID(Event event) throws OXException {
        /*
         * check common folder permissions for events with a static parent folder (public folders)
         */
        if (null != event.getFolderId()) {
            CalendarFolder folder = findFolder(getVisibleFolders(), event.getFolderId());
            if (null != folder) {
                int readPermission = folder.getOwnPermission().getReadPermission();
                if (canReadEvent(event, readPermission)) {
                    return event.getFolderId();
                }
            }
            return null;
        }
        /*
         * prefer user's personal folder if user is attendee
         */
        Attendee ownAttendee = find(event.getAttendees(), session.getUserId());
        if (null != ownAttendee) {
            return ownAttendee.getFolderId();
        }
        /*
         * choose the most appropriate attendee folder, otherwise
         */
        CalendarFolder chosenFolder = chooseAttendeeFolder(event, event.getAttendees());
        if (null != chosenFolder) {
            return chosenFolder.getId();
        }
        /*
         * lookup to which private or shared folders the event belongs to and retry to get an attendee folder
         */
        List<String> possibleFolderIds = new LinkedList<String>();
        for (CalendarFolder calendarFolder : getVisibleFolders()) {
            if (false == PublicType.getInstance().equals(calendarFolder.getType())) {
                possibleFolderIds.add(calendarFolder.getId());
            }
        }
        chosenFolder = chooseAttendeeFolder(event, storage.getAttendeeStorage().loadAttendees(event.getId(), possibleFolderIds, ATTENDEE_FOLDER_SEARCH));
        return null == chosenFolder ? null : chosenFolder.getId();
    }

    private static final AttendeeField[] ATTENDEE_FOLDER_SEARCH = { AttendeeField.FOLDER_ID };

    private boolean canReadEvent(Event event, int readPermission) {
        return Permission.READ_ALL_OBJECTS <= readPermission || Permission.READ_OWN_OBJECTS == readPermission && matches(event.getCreatedBy(), session.getUserId());
    }

    private CalendarFolder chooseAttendeeFolder(Event event, List<Attendee> attendees) throws OXException {
        for (Attendee attendee : attendees) {
            CalendarFolder folder = findFolder(getVisibleFolders(), attendee.getFolderId());
            if (null != folder) {
                int readPermission = folder.getOwnPermission().getReadPermission();
                if (canReadEvent(event, readPermission)) {
                    return folder; // rely on sorting order of visible folders
                }
            }
        }
        return null;
    }

    /**
     * Searches a userized folder in a collection of folders by its numerical identifier.
     *
     * @param folders The folders to search
     * @param id The identifier of the folder to lookup
     * @return The matching folder, or <code>null</code> if not found
     */
    private static CalendarFolder findFolder(Collection<CalendarFolder> folders, String id) {
        if (null != folders && null != id) {
            for (CalendarFolder folder : folders) {
                if (id.equals(folder.getId())) {
                    return folder;
                }
            }
        }
        return null;
    }

}
