/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.chronos.ical;

import java.util.ArrayList;
import java.util.List;
import com.openexchange.exception.OXException;

/**
 * {@link ComponentUtils}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 * @since v7.10.0
 */
public class ComponentUtils {

    /**
     * Collects all warnings from multiple imported components.
     *
     * @param importedComponents The imported components to collect the warnings for
     * @return The collected warnings, or an empty list if there are none
     */
    public static <T> List<OXException> collectWarnings(List<T> importedComponents) {
        List<OXException> warnings = new ArrayList<OXException>();
        if (null != importedComponents) {
            for (T importedComponent : importedComponents) {
                if (ImportedComponent.class.isInstance(importedComponent)) {
                    List<OXException> warningsList = ((ImportedComponent) importedComponent).getWarnings();
                    if (null != warningsList) {
                        warnings.addAll(warningsList);
                    }
                }
            }
        }
        return warnings;
    }

    /**
     * Collects all warnings from multiple imported components.
     *
     * @param ownWarnings The component's own warnings
     * @param subComponents The sub-components to add the warnings from
     * @return The collected warnings, or an empty list if there are none
     */
    public static <T> List<OXException> collectWarnings(List<OXException> ownWarnings, List<T> subComponents) {
        List<OXException> warnings = new ArrayList<OXException>();
        if (null != ownWarnings) {
            warnings.addAll(ownWarnings);
        }
        if (null != subComponents) {
            for (T importedComponent : subComponents) {
                if (ImportedComponent.class.isInstance(importedComponent)) {
                    List<OXException> warningsList = ((ImportedComponent) importedComponent).getWarnings();
                    if (null != warningsList) {
                        warnings.addAll(warningsList);
                    }
                }
            }
        }
        return warnings;
    }

    /**
     * Initializes a new {@link ComponentUtils}.
     */
    private ComponentUtils() {
        super();
    }

}
