/*
* @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
* @license AGPL-3.0
*
* This code is free software: you can redistribute it and/or modify
* it under the terms of the GNU Affero General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU Affero General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public License
* along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
* 
* Any use of the work other than as authorized under this license or copyright law is prohibited.
*
*/

package com.openexchange.chronos.alarm.mail.notification;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Locale;
import java.util.TimeZone;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.chronos.Event;
import com.openexchange.chronos.ParticipationStatus;
import com.openexchange.chronos.Transp;
import com.openexchange.chronos.common.CalendarUtils;
import com.openexchange.chronos.scheduling.changes.MessageContext;
import com.openexchange.chronos.scheduling.changes.Sentence;
import com.openexchange.chronos.scheduling.changes.Sentence.ArgumentType;
import com.openexchange.chronos.scheduling.changes.SentenceFactory;
import com.openexchange.chronos.scheduling.common.Messages;
import com.openexchange.chronos.scheduling.common.description.TypeWrapper;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.exception.OXExceptions;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.notify.hostname.HostnameService;
import com.openexchange.html.HtmlService;
import com.openexchange.html.tools.HTMLUtils;
import com.openexchange.regional.RegionalSettings;
import com.openexchange.server.ServiceLookup;

/**
 * {@link LabelHelper}
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class LabelHelper {

    private final Context ctx;
    private final HTMLUtils html;
    private TimeZone timezone;
    private final ServiceLookup services;
    private final Pattern patternSlashFixer;
    private static final String fallbackHostname;

    private static final Logger LOG = LoggerFactory.getLogger(LabelHelper.class);

    protected final AlarmNotificationMail mail;
    protected final TypeWrapper wrapper;
    protected final Locale locale;

    static {
        String fbHostname;
        try {
            fbHostname = InetAddress.getLocalHost().getCanonicalHostName();
        } catch (UnknownHostException e) {
            fbHostname = "localhost";
        }
        fallbackHostname = fbHostname;
    }

    public LabelHelper(final TimeZone timezone, final AlarmNotificationMail mail, final Locale locale, final Context ctx, final TypeWrapper wrapper, final ServiceLookup services) throws OXException {
        super();
        this.services = services;
        this.mail = mail;
        this.locale = locale;
        this.ctx = ctx;
        this.wrapper = wrapper;
        this.html = new HTMLUtils(services.getService(HtmlService.class));
        this.timezone = timezone;
        if (timezone == null) {
            this.timezone = TimeZone.getDefault(); // Fallback
        }

        patternSlashFixer = Pattern.compile("^//+|[^:]//+");
        if (this.mail == null || this.mail.getEvent() == null) {
            throw OXExceptions.general("Mandatory field mail/event missing.");
        }
    }

    public String getShowAs() {
        final Event event = mail.getEvent();

        if (event.getTransp() != null && Transp.TRANSPARENT.equals(event.getTransp().getValue())) {
            return getSentence(Messages.FREE).getMessage(wrapper.getFormat(), locale, null, null);
        }
        return getSentence(Messages.RESERVERD).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getShowAsClass() {
        final Event event = mail.getEvent();
        if (event.getTransp() != null && event.getTransp().getValue() != null && Transp.TRANSPARENT.equals(event.getTransp().getValue())) {
            return "free";
        }
        return "reserved";
    }

    public String getNoteAsHTML() {
        final String note = mail.getEvent().getDescription();
        if (note == null) {
            return "";
        }
        return html.htmlFormat(note);
    }

    public String getDirectLink() {
        if (mail.getRecipient() == null || mail.getRecipient().isExternal() || mail.getRecipient().isResource()) {
            return null;
        }
        final ConfigurationService config = services.getService(ConfigurationService.class);
        final String template = patternSlashFixer.matcher(config.getProperty("object_link", "https://[hostname]/[uiwebpath]#m=[module]&i=[object]&f=[folder]")).replaceAll("/");
        String webpath = config.getProperty("com.openexchange.UIWebPath", "/appsuite/");
        if (webpath.startsWith("/")) {
            webpath = webpath.substring(1, webpath.length());
        }
        final String objectId = (mail.getEvent() != null) ? mail.getEvent().getId() : null;
        final String module = "calendar";
        String folder = mail.getRecipient().getFolderId();
        int recipientId = mail.getRecipient().getIdentifier();
        if (folder == null) {
            try {
                folder = CalendarUtils.getFolderView(mail.getEvent(), recipientId);
            } catch (OXException e) {
                LOG.error("Unable to generate Link. Folder Id for user {} can't be found.", Integer.valueOf(recipientId), e);
                return "";
            }
        }

        String hostname = null;

        final HostnameService hostnameService = services.getOptionalService(HostnameService.class);
        if (hostnameService != null) {
            if (null != mail.getRecipient().getUser() && mail.getRecipient().getUser().isGuest()) {
                hostname = hostnameService.getGuestHostname(recipientId, ctx.getContextId());
            } else {
                hostname = hostnameService.getHostname(recipientId, ctx.getContextId());
            }
        }

        if (hostname == null) {
            hostname = fallbackHostname;
        }

        if (objectId == null || folder == null) {
            LOG.error("Unable to generate Link. Either Object Id ({}) or Folder Id ({}) is null.", objectId, folder, new Throwable());
            return "";
        }

        return template.replaceAll("\\[hostname\\]", hostname).replaceAll("\\[uiwebpath\\]", webpath).replaceAll("\\[module\\]", module).replaceAll("\\[object\\]", objectId).replaceAll("\\[folder\\]", folder);
    }

    public String getAttachmentNote() {
        if (mail.getEvent().getAttachments() == null || mail.getEvent().getAttachments().isEmpty() || mail.getRecipient().isExternal()) {
            return "";
        }
        return getSentence(Messages.HAS_ATTACHMENTS).add(getDirectLink(), ArgumentType.REFERENCE).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getWhenLabel() {
        return getSentence(Messages.LABEL_WHEN).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getWhereLabel() {
        return getSentence(Messages.LABEL_WHERE).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getParticipantsLabel() {
        return getSentence(Messages.LABEL_PARTICIPANTS).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getResourcesLabel() {
        return getSentence(Messages.LABEL_RESOURCES).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getDetailsLabel() {
        return getSentence(Messages.LABEL_DETAILS).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getShowAsLabel() {
        return getSentence(Messages.LABEL_SHOW_AS).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getCreatedLabel() {
        return getSentence(Messages.LABEL_CREATED).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getDirectLinkLabel() {
        return getSentence(Messages.LINK_LABEL).getMessage(wrapper.getFormat(), locale, null, null);
    }

    public String getTimezoneInfo() {
        return getSentence(Messages.TIMEZONE).add(timezone.getDisplayName(locale), ArgumentType.EMPHASIZED).getMessage(wrapper.getFormat(), locale, null, null);
    }

    Sentence getSentence(String message) {
        SentenceFactory factory = services.getService(SentenceFactory.class);
        if (null == factory) {
            return new Sentence() {

                @Override
                public String getMessage(String format, Locale locale, TimeZone timeZone, RegionalSettings regionalSettings) {
                    return "";
                }

                @Override
                public String getMessage(MessageContext messageContext) {
                    return "";
                }

                @Override
                public Sentence addStatus(ParticipationStatus status) {
                    return this;
                }

                @Override
                public Sentence add(Object argument, ArgumentType type, Object... extra) {
                    return this;
                }
            };
        }
        return factory.create(message);
    }
}
