/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.authentication.application.impl.api;

import java.util.Map;
import com.google.common.collect.ImmutableMap;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXActionServiceFactory;
import com.openexchange.exception.OXException;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.servlet.AjaxExceptionCodes;

/**
 * {@link AppPasswordActionFactory} initializes the available actions for the Application Specific Factory
 * API calls.
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v7.10.4
 */
public class AppPasswordActionFactory implements AJAXActionServiceFactory {

    public static final String MODULE_PATH = "appPasswords";

    private final Map<String, AJAXActionService> actions;

    /**
     * Initializes a new {@link AppPasswordActionFactory}. Sets the available actions
     * 
     * @param services The service lookup
     */
    public AppPasswordActionFactory(ServiceLookup services) {
        super();
        ImmutableMap.Builder<String, AJAXActionService> actions = ImmutableMap.builderWithExpectedSize(4);
        actions.put("add", new AddAppPassword(services));       // Add a new password
        actions.put("remove", new RemoveAppPassword(services)); // Remove existing password
        actions.put("list", new ListApplicationPassword(services));     // List current passwords
        actions.put("getApps", new ListApplications(services));
        this.actions = actions.build();
    }

    @Override
    public AJAXActionService createActionService(String action) throws OXException {
        AJAXActionService actionService = actions.get(action);
        if (actionService == null) {
            throw AjaxExceptionCodes.UNKNOWN_ACTION_IN_MODULE.create(action, MODULE_PATH);
        }
        return actionService;
    }
}
