/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 Open-Xchange GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.ant.tasks;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import com.openexchange.obs.api.Architecture;
import com.openexchange.obs.api.BlockMode;
import com.openexchange.obs.api.BuildServiceClient;
import com.openexchange.obs.api.BuildServiceException;
import com.openexchange.obs.api.BuildTrigger;
import com.openexchange.obs.api.DistroNaming;
import com.openexchange.obs.api.Path;
import com.openexchange.obs.api.Project;
import com.openexchange.obs.api.ProjectReference;
import com.openexchange.obs.api.Repository;
import com.openexchange.obs.api.RepositoryReference;

/**
 * Creates a new project on the open build service. First a check is done if the project already exists. If the project already exists
 * nothing is done.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class CreateProject extends Task {

    private String url;
    private String login;
    private String password;
    private String projectName;
    private String obsConfig;
    private String buildTrigger;
    private String blockMode;
    private String[] repositories;
    private String arch;
    private String[] disabledRepositories;
    private String[] dependingProjectNames;

    public CreateProject() {
        super();
    }

    public void setUrl(final String url) {
        this.url = url;
    }

    public void setLogin(final String login) {
        this.login = login;
    }

    public void setPassword(final String password) {
        this.password = password;
    }

    public void setProjectName(final String projectName) {
        this.projectName = projectName;
    }

    public void setRepositories(final String repositories){
        if (repositories.length() != 0){
            this.repositories = repositories.split(",");
        }
    }

    public final void setArch(String arch) {
        this.arch = arch;
    }

    public void setDisabledRepositories(String disabledRepositories) {
        if (disabledRepositories.length() != 0) {
            this.disabledRepositories = disabledRepositories.split(",");
        }
    }

    public void setDependingProjectNames(final String dependingProjectNames){
        if (dependingProjectNames.length() != 0) {
            this.dependingProjectNames = dependingProjectNames.split(",");
        }
    }

    public void setObsConfig(final String obsConfig){
        if (obsConfig.length() != 0) {
            this.obsConfig = obsConfig;
        }
    }

    public void setBuildTrigger(String buildTrigger) {
        this.buildTrigger = buildTrigger;
    }

    public void setBlockMode(String blockMode) {
        this.blockMode = blockMode;
    }

    @Override
    public void execute() throws BuildException {
        if (projectName == null) {
            throw new BuildException("Please define project.");
        }
        if (repositories == null) {
            throw new BuildException("Please define repositories.");
        }
        if (null == arch) {
            throw new BuildException("Please define architecture.");
        }
        if (null == buildTrigger) {
            buildTrigger = getProject().getProperty("obsBuildTrigger");
        }
        if (null == blockMode) {
            blockMode = getProject().getProperty("obsBlockMode");
        }
        //dependingProjectNames is allowed to be null
        final BuildServiceClient client;
        try {
            Collection<Architecture> architecture = Architecture.parseArchitecture(arch);
            client = new BuildServiceClient(url, login, password);
            final Project obsProject = createProject(projectName, login, repositories, architecture, disabledRepositories, dependingProjectNames, obsConfig, parseRebuild(buildTrigger), parseBlock(blockMode), new PropertyGetter() {
                public String getProperty(String name) {
                    return getProject().getProperty(name);
                }
            });
            if (!client.existsProject(obsProject)) {
                log("Creating project \"" + obsProject.getName() + "\" on build service.", org.apache.tools.ant.Project.MSG_INFO);
                client.createProject(obsProject);
            } else if (!client.getProjectConfig(obsProject.getName()).equals(obsProject.getBuildConfiguration())){
                log("Project \"" + obsProject.getName() + "\" already exists on build service but has different config.", org.apache.tools.ant.Project.MSG_INFO);
                log("Project \"" + obsProject.getName() + "\" updated on build service.", org.apache.tools.ant.Project.MSG_INFO);
                client.createProject(obsProject);
            } else {
                log("Project \"" + obsProject.getName() + "\" already exists on build service.", org.apache.tools.ant.Project.MSG_INFO);

                Project currentProject = client.getRepositoryList(projectName);
                final Set<Repository> currentRepositories = new HashSet<Repository>();
                System.out.println("currently used repositories on obs: ");
                for (Repository r : currentProject.getRepositories()) {
                    boolean disabled = false;
                    for (String disabledRepo : currentProject.getDisabledRepositories()) {
                        if (disabledRepo.equals(r.getName())) {
                            disabled = true;
                            break;
                        }
                    }
                    if (!disabled) {
                        System.out.println(r.toString());
                        currentRepositories.add(r);
                    }
                }

                System.out.println("needed repositories: ");
                final Set<Repository> neededRepositories = new HashSet<Repository>();
                for (Repository r : obsProject.getRepositories()) {
                    boolean disabled = false;
                    for (String disabledRepo : obsProject.getDisabledRepositories()) {
                        if (disabledRepo.equals(r.getName())) {
                            disabled = true;
                            break;
                        }
                    }
                    if (!disabled) {
                        System.out.println(r.toString());
                        neededRepositories.add(r);
                    }
                }
                if (!currentRepositories.equals(neededRepositories)) {
                    log("Project \"" + obsProject.getName() + "\" updated on build service.", org.apache.tools.ant.Project.MSG_INFO);
                    client.createProject(obsProject);
                }
            }
        } catch (final BuildServiceException e) {
            throw new BuildException(e.getMessage(), e);
        }
    }

    private static final BuildTrigger parseRebuild(String obsBuildTrigger) throws BuildServiceException {
        if (null == obsBuildTrigger || 0 == obsBuildTrigger.length()) {
            return BuildTrigger.LOCAL;
        }
        return BuildTrigger.parse(obsBuildTrigger);
    }

    private static final BlockMode parseBlock(String obsBlockMode) throws BuildServiceException {
        if (null == obsBlockMode || 0 == obsBlockMode.length()) {
            return BlockMode.DEFAULT;
        }
        return BlockMode.parse(obsBlockMode);
    }

    private static Project createProject(String projectName, String user, String[] repositories, Collection<Architecture> arch, String[] disabledRepositories, String[] dependingProjectNames, String obsConfig, BuildTrigger rebuild, BlockMode block, PropertyGetter propGetter) throws BuildServiceException {
        final Project project = new Project(projectName, projectName, user);
        if (null != disabledRepositories) {
            for (final String s : disabledRepositories) {
                project.addDisabledRepository(s);
            }
        }
        for (final String name : repositories) {
            if (null == dependingProjectNames) {
                String base = propGetter.getProperty(name + "-Base");
                if (null == base) {
                    project.addRepository(new Repository(name, Path.standard(DistroNaming.parse(name)), arch, rebuild, block));
                } else {
                    int pos = base.indexOf('/');
                    if (-1 == pos) {
                        project.addRepository(new Repository(name, new Path(new ProjectReference(base), RepositoryReference.STANDARD), arch, rebuild, block));
                    } else {
                        String repository = base.substring(pos + 1);
                        base = base.substring(0, pos);
                        project.addRepository(new Repository(name, new Path(new ProjectReference(base), new RepositoryReference(repository)), arch, rebuild, block));
                    }
                }
            } else {
                List<Path> paths = new LinkedList<Path>();
                for (String dependency : dependingProjectNames) {
                    paths.add(new Path(new ProjectReference(dependency), new RepositoryReference(name)));
                }
                project.addRepository(new Repository(name, paths, arch, rebuild, block));
            }
        }
        project.setBuildConfiguration((obsConfig != null ? obsConfig : "Prefer: ") + "\n" + DEFAULT_CONFIGURATION);
        return project;
    }

    private static final String DEFAULT_CONFIGURATION =
        "Macros:\n" +
        "%vendor Open-Xchange\n";
}
