/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2016 Open-Xchange GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.ant.data;

import static com.openexchange.osgi.Tools.calculateModulesByName;
import static com.openexchange.osgi.ant.Tools.readClasspathModules;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.types.DataType;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.Resource;
import org.apache.tools.ant.types.ResourceCollection;
import org.apache.tools.ant.types.resources.FileResource;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import com.openexchange.build.utils.Tools;
import com.openexchange.osgi.data.AbstractModule;

/**
 * Reads and Eclipse launcher configuration and maps the listed bundles to files of a bundles directory.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public class LauncherBundles extends DataType implements ResourceCollection {

    private File launcher;
    private Path classpath;

    public LauncherBundles() {
        super();
    }

    public void setLauncher(File launcher) {
        this.launcher = launcher;
    }

    public Path createClasspath() {
        classpath = new Path(getProject());
        return classpath;
    }

    private List<String> parseLauncherBundles() throws Exception {
        if (null == launcher) {
            throw new BuildException("You need to define the launcher attribute.");
        }
        final Document document = com.openexchange.ant.Tools.readXML(launcher);
        final Element launchConfigurationElement = document.getDocumentElement();
        final NodeList childs = launchConfigurationElement.getChildNodes();
        final List<String> retval = new ArrayList<String>();
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("stringAttribute".equals(node.getNodeName())) {
                final NamedNodeMap attributes = node.getAttributes();
                final Node key = attributes.getNamedItem("key");
                final Node value = attributes.getNamedItem("value");
                if (null == key || null == value) {
                    continue;
                }
                if ("target_bundles".equals(key.getNodeValue())) {
                    retval.addAll(parseValue(value.getNodeValue()));
                } else if ("workspace_bundles".equals(key.getNodeValue())) {
                    retval.addAll(parseValue(value.getNodeValue()));
                }
            }
        }
        return retval;
    }

    private static List<String> parseValue(String value) throws Exception {
        final List<String> retval = new ArrayList<String>();
        for (String tmp : Tools.split(value, ',')) {
            int pos = tmp.indexOf('@');
            if (pos <= 0) {
                throw new Exception("Can not parse \"" + tmp + "\".");
            }
            retval.add(tmp.substring(0, pos));
        }
        return retval;
    }

    private List<Resource> bundles = null;

    private List<Resource> getBundles() {
        if (null == classpath) {
            throw new BuildException("You need to define the nested classpath attribute.");
        }
        if (null == bundles) {
            log("using classpath: " + classpath, Project.MSG_DEBUG);
            final List<AbstractModule> classpathModules;
            try {
                classpathModules = readClasspathModules(classpath);
            } catch (IOException e) {
                throw new BuildException(e.getMessage(), e);
            }
            final Map<String, AbstractModule> modulesByName = calculateModulesByName(classpathModules);
            bundles = new ArrayList<Resource>();
            List<String> launcherBundles;
            try {
                launcherBundles = parseLauncherBundles();
            } catch (DOMException e) {
                throw new BuildException(e);
            } catch (Exception e) {
                throw new BuildException(e);
            }
            for (final String bundle : launcherBundles) {
                AbstractModule module = modulesByName.get(bundle);
                if ("com.openexchange.xerces.sun".equals(bundle) && null == module) {
                    // Workaround for IBM Java
                    module = modulesByName.get("com.openexchange.xerces.ibm");
                }
                if (null == module) {
                    throw new BuildException("Can not find bundle \"" + bundle + "\".");
                }
                bundles.add(new FileResource(module.getFile()));
            }
        }
        return bundles;
    }

    public Iterator<Resource> iterator() {
        return getBundles().iterator();
    }

    public int size() {
        return getBundles().size();
    }

    public boolean isFilesystemOnly() {
        return true;
    }
}
