/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package gnu.trove.map.hash;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.StampedLock;
import gnu.trove.function.TObjectFunction;
import gnu.trove.iterator.TIntObjectIterator;
import gnu.trove.map.TIntObjectMap;
import gnu.trove.procedure.TIntObjectProcedure;
import gnu.trove.procedure.TIntProcedure;
import gnu.trove.procedure.TObjectProcedure;

/**
 * {@link ConcurrentTIntObjectHashMap} - A concurrent {@link TIntObjectHashMap} implementation using a {@link ReadWriteLock} instance.
 * <p>
 * An open addressed Map implementation for int keys and Object values.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public final class ConcurrentTIntObjectHashMap<V> extends TIntObjectHashMap<V> implements Cloneable {

    private static class Pair<V> {

        final int key;
        final V value;

        Pair(int key, V value) {
            super();
            this.key = key;
            this.value = value;
        }
    }

    // -------------------------------------------------------------------------------------------------------------------------------------

    private final TIntObjectHashMap<V> map;
    private final StampedLock lock;

    /**
     * Initializes a new {@link ConcurrentTIntObjectHashMap}.
     */
    public ConcurrentTIntObjectHashMap() {
        super();
        map = new TIntObjectHashMap<V>();
        lock = new StampedLock();
    }

    /**
     * Initializes a new {@link ConcurrentTIntObjectHashMap}.
     *
     * @param initialCapacity used to find a prime capacity for the table.
     * @param loadFactor used to calculate the threshold over which rehashing takes place.
     */
    public ConcurrentTIntObjectHashMap(final int initialCapacity, final float loadFactor) {
        super();
        map = new TIntObjectHashMap<V>(initialCapacity, loadFactor);
        lock = new StampedLock();
    }

    /**
     * Initializes a new {@link ConcurrentTIntObjectHashMap}.
     *
     * @param initialCapacity used to find a prime capacity for the table.
     */
    public ConcurrentTIntObjectHashMap(final int initialCapacity) {
        super();
        map = new TIntObjectHashMap<V>(initialCapacity);
        lock = new StampedLock();
    }

    /**
     * Copy constructor.
     */
    protected ConcurrentTIntObjectHashMap(ConcurrentTIntObjectHashMap<V> another) {
        super();
        map = new TIntObjectHashMap<V>(another.map);
        lock = new StampedLock();
    }

    @Override
    public ConcurrentTIntObjectHashMap<V> clone() {
        return new ConcurrentTIntObjectHashMap<V>(this);
    }

    /**
     * Gets the read-write lock associated with this map
     *
     * @return The read-write lock
     */
    public ReadWriteLock getReadWriteLock() {
        return lock.asReadWriteLock();
    }

    /**
     * Copies entries to specified map
     *
     * @param map The map to copy to
     */
    public void copySafeTo(final TIntObjectMap<V> map) {
        if (null == map) {
            return;
        }
        long stamp = lock.readLock();
        try {
            map.putAll(this.map);
        } finally {
            lock.unlockRead(stamp);
        }
    }

    @Override
    public TIntObjectIterator<V> iterator() {
        long stamp = lock.tryOptimisticRead();
        List<Pair<V>> l;
        {
            final List<Pair<V>> entries = new ArrayList<ConcurrentTIntObjectHashMap.Pair<V>>(map.size());
            map.forEachEntry(new TIntObjectProcedure<V>() {

                @Override
                public boolean execute(int a, V b) {
                    entries.add(new Pair<V>(a, b));
                    return true;
                }
            });
            l = entries;
        }

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                final List<Pair<V>> newEntries = new ArrayList<ConcurrentTIntObjectHashMap.Pair<V>>(map.size());
                map.forEachEntry(new TIntObjectProcedure<V>() {

                    @Override
                    public boolean execute(int a, V b) {
                        newEntries.add(new Pair<V>(a, b));
                        return true;
                    }
                });
                l = newEntries;
            } finally {
                lock.unlockRead(stamp);
            }
        }

        List<Pair<V>> list = l;
        return new  TIntObjectIterator<V>() {

            private int index = 0;

            @Override
            public void advance() {
                index++;
            }

            @Override
            public boolean hasNext() {
                return index < list.size();
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException("ConcurrentTIntObjectHashMap.TIntObjectIterator().remove()");
            }

            @Override
            public int key() {
                return list.get(index).key;
            }

            @Override
            public V value() {
                return list.get(index).value;
            }

            @Override
            public V setValue(V val) {
                throw new UnsupportedOperationException("ConcurrentTIntObjectHashMap.TIntObjectIterator().setValue()");
            }

        };
    }

    @Override
    public boolean isEmpty() {
        long stamp = lock.tryOptimisticRead();
        boolean empty = map.isEmpty();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                empty = map.isEmpty();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return empty;
    }

    @Override
    public boolean contains(final int val) {
        long stamp = lock.tryOptimisticRead();
        boolean contains = map.contains(val);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                contains = map.contains(val);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return contains;
    }

    @Override
    public int size() {
        long stamp = lock.tryOptimisticRead();
        int size = map.size();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                size = map.size();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return size;
    }

    @Override
    public boolean forEach(final TIntProcedure procedure) {
        long stamp = lock.readLock();
        try {
            return map.forEach(procedure);
        } finally {
            lock.unlockRead(stamp);
        }
    }

    @Override
    public V put(final int key, final V value) {
        long stamp = lock.writeLock();
        try {
            return map.put(key, value);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public void ensureCapacity(final int desiredCapacity) {
        long stamp = lock.writeLock();
        try {
            map.ensureCapacity(desiredCapacity);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public V putIfAbsent(final int key, final V value) {
        long stamp = lock.writeLock();
        try {
            return map.putIfAbsent(key, value);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public void compact() {
        long stamp = lock.writeLock();
        try {
            map.compact();
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    private final TIntObjectProcedure<V> PUT_ALL_PROC = new TIntObjectProcedure<V>() {
        @Override
        public boolean execute(final int key, final V value) {
            put( key, value );
            return true;
        }
    };

    @Override
    public void putAll(final TIntObjectMap<? extends V> map) {
        long stamp = lock.writeLock();
        try {
            map.forEachEntry( PUT_ALL_PROC );
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public void setAutoCompactionFactor(final float factor) {
        long stamp = lock.writeLock();
        try {
            map.setAutoCompactionFactor(factor);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public V get(final int key) {
        long stamp = lock.tryOptimisticRead();
        V v = map.get(key);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                v = map.get(key);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return v;
    }

    @Override
    public float getAutoCompactionFactor() {
        long stamp = lock.tryOptimisticRead();
        float f = map.getAutoCompactionFactor();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                f = map.getAutoCompactionFactor();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return f;
    }

    @Override
    public void clear() {
        long stamp = lock.writeLock();
        try {
            map.clear();
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public V remove(final int key) {
        long stamp = lock.writeLock();
        try {
            return map.remove(key);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public boolean equals(final Object other) {
        long stamp = lock.tryOptimisticRead();
        boolean equals = map.equals(other);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                equals = map.equals(other);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return equals;
    }

    @Override
    public int hashCode() {
        long stamp = lock.tryOptimisticRead();
        int h = map.hashCode();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                h = map.hashCode();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return h;
    }

    @Override
    public Object[] values() {
        long stamp = lock.tryOptimisticRead();
        Object[] values = map.values();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                values = map.values();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return values;
    }

    @Override
    public V[] values(final V[] a) {
        long stamp = lock.tryOptimisticRead();
        V[] values = map.values(a);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                values = map.values(a);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return values;
    }

    @Override
    public int[] keys() {
        long stamp = lock.tryOptimisticRead();
        int[] keys = map.keys();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                keys = map.keys();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return keys;
    }

    @Override
    public int[] keys(final int[] a) {
        long stamp = lock.tryOptimisticRead();
        int[] keys = map.keys(a);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                keys = map.keys(a);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return keys;
    }

    @Override
    public boolean containsValue(final Object val) {
        long stamp = lock.tryOptimisticRead();
        boolean contains = map.containsValue(val);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                contains = map.containsValue(val);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return contains;
    }

    @Override
    public boolean containsKey(final int key) {
        long stamp = lock.tryOptimisticRead();
        boolean contains = map.containsKey(key);

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                contains = map.containsKey(key);
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return contains;
    }

    @Override
    public boolean forEachKey(final TIntProcedure procedure) {
        long stamp = lock.readLock();
        try {
            return map.forEachKey(procedure);
        } finally {
            lock.unlockRead(stamp);
        }
    }

    @Override
    public boolean forEachValue(final TObjectProcedure<? super V> procedure) {
        long stamp = lock.readLock();
        try {
            return map.forEachValue(procedure);
        } finally {
            lock.unlockRead(stamp);
        }
    }

    @Override
    public boolean forEachEntry(final TIntObjectProcedure<? super V> procedure) {
        long stamp = lock.readLock();
        try {
            return map.forEachEntry(procedure);
        } finally {
            lock.unlockRead(stamp);
        }
    }

    @Override
    public boolean retainEntries(final TIntObjectProcedure<? super V> procedure) {
        long stamp = lock.writeLock();
        try {
            return map.retainEntries(procedure);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public void transformValues(final TObjectFunction<V, V> function) {
        long stamp = lock.writeLock();
        try {
            map.transformValues(function);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public void writeExternal(final ObjectOutput out) throws IOException {
        long stamp = lock.readLock();
        try {
            map.writeExternal(out);
        } finally {
            lock.unlockRead(stamp);
        }
    }

    @Override
    public void readExternal(final ObjectInput in) throws IOException, ClassNotFoundException {
        long stamp = lock.writeLock();
        try {
            map.readExternal(in);
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    @Override
    public String toString() {
        long stamp = lock.tryOptimisticRead();
        String s = map.toString();

        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                s = map.toString();
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return s;
    }

}
