/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.filestore.s3.internal;

import static com.openexchange.filestore.s3.internal.AbortIfNotConsumedInputStream.closeContentStream;
import static com.openexchange.filestore.s3.internal.S3ExceptionCode.wrap;
import static com.openexchange.java.Autoboxing.L;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.UUID;
import javax.servlet.http.HttpServletResponse;
import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.CopyPartRequest;
import com.amazonaws.services.s3.model.CopyPartResult;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.amazonaws.services.s3.model.MultiObjectDeleteException.DeleteError;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazonaws.services.s3.model.UploadPartResult;
import com.openexchange.exception.OXException;
import com.openexchange.filestore.FileStorageCodes;
import com.openexchange.filestore.s3.internal.client.S3FileStorageClient;
import com.openexchange.filestore.utils.TempFileHelper;
import com.openexchange.java.Sets;
import com.openexchange.java.Streams;
import com.openexchange.java.Strings;
import com.openexchange.java.util.UUIDs;
import com.openexchange.tools.arrays.Arrays;

/**
 * {@link PlainS3FileStorage} - The plain S3 file storage that does not use a database-backed chunk storage.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public class PlainS3FileStorage extends AbstractS3FileStorage {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(PlainS3FileStorage.class);

    /**
     * Initializes a new {@link PlainS3FileStorage}.
     *
     * @param uri The URI that fully qualifies this file storage
     * @param prefix The prefix to use; e.g. <code>"1337ctxstore"</code>
     * @param bucketName The bucket name to use
     * @param client The file storage client
     */
    public PlainS3FileStorage(URI uri, String prefix, String bucketName, S3FileStorageClient client) {
        super(uri, prefix, bucketName, client);
        LOG.debug("S3 file storage initialized for \"{}/{}{}\"", bucketName, prefix, DELIMITER);
    }

    @Override
    public String saveNewFile(InputStream input) throws OXException {
        return saveNewFile(input, true);
    }

    /**
     * Saves specified content as new S3 object.
     *
     * @param input The content for the S3 object
     * @param considerSpooling Whether to consider spooling given content to a temporary file or not
     * @return The S3 object identifier
     * @throws OXException If saving new S3 object fails
     */
    private String saveNewFile(InputStream input, boolean considerSpooling) throws OXException {
        /*
         * perform chunked upload as needed
         */
        S3ChunkedUpload chunkedUpload = null;
        S3UploadChunk chunk = null;
        File tmpFile = null;
        try {
            /*
             * spool to file
             */
            if (considerSpooling && !(input instanceof ByteArrayInputStream) && (!(input instanceof FileInputStream))) {
                Optional<File> optionalTempFile = TempFileHelper.getInstance().newTempFile();
                if (optionalTempFile.isPresent()) {
                    tmpFile = optionalTempFile.get();
                    input = Streams.transferToFileAndCreateStream(input, tmpFile);
                }
            }
            /*
             * proceed
             */
            String key = generateKey(true);
            chunkedUpload = new S3ChunkedUpload(input, client.getEncryptionConfig().isClientEncryptionEnabled(), client.getChunkSize());
            chunk = chunkedUpload.next();
            if (false == chunkedUpload.hasNext()) {
                /*
                 * whole file fits into buffer (this includes a zero byte file), upload directly
                 */
                uploadSingle(key, chunk);
            } else {
                /*
                 * upload in multipart chunks to provide the correct content length
                 */
                long contentLength = 0L;
                InitiateMultipartUploadRequest initiateMultipartUploadRequest = new InitiateMultipartUploadRequest(bucketName, key).withObjectMetadata(prepareMetadataForSSE(new ObjectMetadata()));
                String uploadID = withRetry(c -> c.initiateMultipartUpload(initiateMultipartUploadRequest).getUploadId());
                boolean completed = false;
                try {
                    Thread currentThread = Thread.currentThread();
                    List<PartETag> partETags = new ArrayList<PartETag>();
                    int partNumber = 1;
                    /*
                     * upload n-1 parts
                     */
                    do {
                        partETags.add(uploadPart(key, uploadID, partNumber++, chunk, false).getPartETag());
                        if (currentThread.isInterrupted()) {
                            throw OXException.general("Upload to S3 aborted");
                        }
                        contentLength += chunk.getSize();
                        chunk = chunkedUpload.next();
                    } while (chunkedUpload.hasNext());
                    /*
                     * upload last part & complete upload
                     */
                    partETags.add(uploadPart(key, uploadID, partNumber++, chunk, true).getPartETag());
                    contentLength += chunk.getSize();
                    withRetry(c -> c.completeMultipartUpload(new CompleteMultipartUploadRequest(bucketName, key, uploadID, partETags)));
                    applyUnencryptedContentLengthIfNeeded(key, contentLength);
                    completed = true;
                } finally {
                    if (false == completed) {
                        try {
                            withVoidRetry(c -> c.abortMultipartUpload(new AbortMultipartUploadRequest(bucketName, key, uploadID)));
                        } catch (AmazonClientException e) {
                            LOG.warn("Error aborting multipart upload", e);
                        }
                    }
                }
            }
            return removePrefix(key);
        } catch (IOException e) {
            throw FileStorageCodes.IOERROR.create(e, e.getMessage());
        } finally {
            Streams.close(chunk, chunkedUpload, input);
            TempFileHelper.deleteQuietly(tmpFile);
        }
    }

    @Override
    public InputStream getFile(String name) throws OXException {
        String key = addPrefix(name);
        S3ObjectInputStream objectContent = null;
        try {
            objectContent = getObject(key).getObjectContent();
            InputStream wrapper = wrapperWithoutRangeSupport(objectContent, key);
            objectContent = null; // Avoid premature closing
            return wrapper;
        } catch (AmazonClientException e) {
            throw wrap(e, key);
        } finally {
            closeContentStream(objectContent);
        }
    }

    @Override
    public InputStream getFile(String name, long offset, long length) throws OXException {
        // Check validity of given offset/length arguments
        long fileSize = getFileSize(name);
        if (offset >= fileSize || (length >= 0 && length > fileSize - offset)) {
            throw FileStorageCodes.INVALID_RANGE.create(L(offset), L(length), name, L(fileSize));
        }

        // Check for 0 (zero) requested bytes
        if (length == 0) {
            return Streams.EMPTY_INPUT_STREAM;
        }

        // Initialize appropriate Get-Object request
        String key = addPrefix(name);
        GetObjectRequest request = new GetObjectRequest(bucketName, key);
        long rangeEnd = (length > 0 ? (offset + length) : fileSize) - 1;
        request.setRange(offset, rangeEnd);

        // Return content stream
        S3ObjectInputStream objectContent = null;
        try {
            objectContent = withRetry(c -> c.getObject(request).getObjectContent());
            long[] range = new long[] { offset, rangeEnd };
            InputStream wrapper = wrapperWithRangeSupport(objectContent, range, key);
            objectContent = null; // Avoid premature closing
            return wrapper;
        } catch (AmazonClientException e) {
            if ((e instanceof AmazonServiceException) && HttpServletResponse.SC_REQUESTED_RANGE_NOT_SATISFIABLE == ((AmazonServiceException) e).getStatusCode()) {
                throw FileStorageCodes.INVALID_RANGE.create(e, L(offset), L(length), name, L(fileSize));
            }
            throw wrap(e, key);
        } finally {
            closeContentStream(objectContent);
        }
    }

    @Override
    public SortedSet<String> getFileList() throws OXException {
        SortedSet<String> files = new TreeSet<String>();
        /*
         * results may be paginated - repeat listing objects as long as result is truncated
         */
        ListObjectsRequest listObjectsRequest = new ListObjectsRequest().withBucketName(bucketName).withDelimiter(String.valueOf(DELIMITER)).withPrefix(new StringBuilder(prefix).append(DELIMITER).toString());
        ObjectListing objectListing;
        do {
            objectListing = withRetry(c -> c.listObjects(listObjectsRequest));
            for (S3ObjectSummary objectSummary : objectListing.getObjectSummaries()) {
                files.add(removePrefix(objectSummary.getKey()));
            }
            listObjectsRequest.setMarker(objectListing.getNextMarker());
        } while (objectListing.isTruncated());
        return files;
    }

    @Override
    public long getFileSize(String name) throws OXException {
        return getContentLength(addPrefix(name));
    }

    @Override
    public String getMimeType(String name) throws OXException {
        //TODO: makes no sense at storage layer
        return getMetadata(addPrefix(name)).getContentType();
    }

    @Override
    public boolean deleteFile(String name) throws OXException {
        String key = addPrefix(name);
        try {
            withVoidRetry(c -> c.deleteObject(bucketName, key));
            return true;
        } catch (AmazonClientException e) {
            throw wrap(e, key);
        }
    }

    @Override
    public Set<String> deleteFiles(String[] names) throws OXException {
        if (null == names || 0 >= names.length) {
            return Collections.emptySet();
        }

        Set<String> notDeleted = new HashSet<String>();
        for (String[] partition : Arrays.partition(names, MAX_NUMBER_OF_KEYS_TO_DELETE)) {
            DeleteObjectsRequest deleteRequest = new DeleteObjectsRequest(bucketName).withKeys(addPrefix(partition));
            try {
                withRetry(c -> c.deleteObjects(deleteRequest));
            } catch (MultiObjectDeleteException e) {
                List<DeleteError> errors = e.getErrors();
                if (null != errors && !errors.isEmpty()) {
                    for (DeleteError error : errors) {
                        notDeleted.add(removePrefix(error.getKey()));
                    }
                }
            } catch (AmazonClientException e) {
                throw wrap(e);
            }
        }
        return notDeleted;
    }

    @Override
    public void remove() throws OXException {
        try {
            /*
             * try and delete all contained files repeatedly
             */
            final int RETRY_COUNT = 10;
            for (int i = 0; i < RETRY_COUNT; i++) {
                try {
                    SortedSet<String> fileList = getFileList();
                    if (null == fileList || fileList.isEmpty()) {
                        return; // no more files found
                    }

                    for (Set<String> partition : Sets.partition(fileList, MAX_NUMBER_OF_KEYS_TO_DELETE)) {
                        withRetry(c -> c.deleteObjects(new DeleteObjectsRequest(bucketName).withKeys(addPrefix(partition))));
                    }
                } catch (MultiObjectDeleteException e) {
                    if (i < RETRY_COUNT - 1) {
                        LOG.warn("Not all files in bucket deleted yet, trying again.", e);
                    } else {
                        throw FileStorageCodes.NOT_ELIMINATED.create(e, "Not all files in bucket deleted after " + i + " tries, giving up.");
                    }
                }
            }
        } catch (OXException e) {
            throw FileStorageCodes.NOT_ELIMINATED.create(e);
        } catch (AmazonClientException e) {
            throw FileStorageCodes.NOT_ELIMINATED.create(wrap(e));
        }
    }

    @Override
    public long appendToFile(InputStream file, String name, long offset) throws OXException {
        /*
         * Get the content length of the object to add the data to
         */
        long contentLength = getContentLength(addPrefix(name));
        if (contentLength != offset) {
            throw FileStorageCodes.INVALID_OFFSET.create(Long.valueOf(offset), name, Long.valueOf(contentLength));
        }

        File tmpFile = null;
        InputStream content = file;
        try {
            /*
             * spool to file
             */
            if (!(content instanceof ByteArrayInputStream) && !(content instanceof FileInputStream)) {
                Optional<File> optionalTempFile = TempFileHelper.getInstance().newTempFile();
                if (optionalTempFile.isPresent()) {
                    tmpFile = optionalTempFile.get();
                    content = Streams.transferToFileAndCreateStream(content, tmpFile);
                }
            }

            /*-
             * Append the data: CopyPart/UploadPartCopy is preferred, because it does not require to download the S3 object for the
             * append operation. But: UploadPartCopy might not be available/possible in all situations
             */
            return isCopyPartAvailable(contentLength) ? appendToFileWithCopyPart(content, name) : appendToFileWithSpooling(content, name);
        } catch (IOException e) {
            throw FileStorageCodes.IOERROR.create(e, e.getMessage());
        } finally {
            Streams.close(content);
            TempFileHelper.deleteQuietly(tmpFile);
        }
    }

    /**
     * Set the new file length of an existing S3 object  (shortens an object)
     * <p>
     * This method uses a <i>UploadPartCopy</i> ({@link CopyPartRequest}) within a multipart upload request in order to create a new temporary S3 object with shortened data.
     * Afterwards the temporary object replaces the original.
     * </p>
     * <p>
     * Due to S3 restrictions, a UploadPartCopy is only possible if the source file, is larger than 5 MB.
     * </p>
     *
     * @param length The new length of object
     * @param name The name of the object
     * @throws OXException
     */
    private void setFileLengthWithCopyPart(long length, String name) throws OXException {
        /*
         * Initialize the multipart upload
         */
        String key = addPrefix(name);
        String tmpKey = generateKey(true);
        InitiateMultipartUploadRequest initiateMultipartUploadRequest = new InitiateMultipartUploadRequest(bucketName, tmpKey).withObjectMetadata(prepareMetadataForSSE(new ObjectMetadata()));
        String uploadID = withRetry(c -> c.initiateMultipartUpload(initiateMultipartUploadRequest).getUploadId());
        boolean uploadCompleted = false;

        try {
            /*
             * Copy the object in parts to a temporary S3 object with the desired length
             */
            long partSize = 5 * 1024 * 1024L;
            long bytePosition = 0;
            int partNum = 1;
            List<PartETag> parts = new ArrayList<PartETag>();
            while (bytePosition < length) {

                // The last part might be smaller than partSize, so check to make sure
                // that lastByte isn't beyond the end of the object.
                long lastByte = Math.min(bytePosition + partSize - 1, length - 1);

                // Copy this part.
                //@formatter:off
                CopyPartRequest copyRequest = new CopyPartRequest()
                    .withSourceBucketName(bucketName)
                    .withSourceKey(key)
                    .withDestinationBucketName(bucketName)
                    .withDestinationKey(tmpKey)
                    .withUploadId(uploadID)
                    .withFirstByte(L(bytePosition))
                    .withLastByte(L(lastByte))
                    .withPartNumber(partNum++);
                //@formatter:on
                CopyPartResult copyPartResult = withRetry(c -> c.copyPart(copyRequest));
                parts.add(copyPartResult.getPartETag());
                bytePosition += partSize;
            }

            /*
             * complete the request
             */
            withRetry(c -> c.completeMultipartUpload(new CompleteMultipartUploadRequest(bucketName, tmpKey, uploadID, parts)));
            uploadCompleted = true;

            /*
             * replace the original
             */
            CopyObjectRequest copyObjectRequest = new CopyObjectRequest(bucketName, tmpKey, bucketName, key);
            ObjectMetadata metadata = prepareMetadataForSSE(new ObjectMetadata());
            copyObjectRequest.setNewObjectMetadata(metadata);
            withRetry(c -> c.copyObject(copyObjectRequest));
            withVoidRetry(c -> c.deleteObject(bucketName, tmpKey));
        } catch (AmazonClientException e) {
            throw wrap(e, key);
        } finally {
            if (uploadCompleted == false) {
                try {
                    withVoidRetry(c -> c.abortMultipartUpload(new AbortMultipartUploadRequest(bucketName, tmpKey, uploadID)));
                } catch (AmazonClientException e) {
                    LOG.warn("Error aborting multipart upload", e);
                }
            }
        }
    }

    /**
     * Set the new file length of an existing S3 object  (shortens an object)
     * <p>
     * This method downloads the existing S3 object and uploads it again, up to the desired new length.
     * </p>
     *
     * @param file The data to append as {@link InputStream}
     * @param name The name of the object to append the data to
     * @return The new size of the modified object
     * @throws OXException
     */
    private void setFileLengthWithSpooling(long length, String name) throws OXException {
        /*
         * copy previous file to temporary file
         */
        String key = addPrefix(name);
        String tempKey = generateKey(true);
        try {
            CopyObjectRequest copyObjectRequest = new CopyObjectRequest(bucketName, key, bucketName, tempKey);
            ObjectMetadata metadata = prepareMetadataForSSE(new ObjectMetadata());
            copyObjectRequest.setNewObjectMetadata(metadata);
            withRetry(c -> c.copyObject(copyObjectRequest));
            /*
             * upload $length bytes from previous file to new current file
             */
            metadata = new ObjectMetadata();
            metadata.setContentLength(length);
            metadata = prepareMetadataForSSE(metadata);
            InputStream inputStream = getFile(tempKey, 0, length);
            try {
                ObjectMetadata meta = metadata;
                withRetry(c -> c.putObject(bucketName, key, inputStream, meta));
            } finally {
                Streams.close(inputStream);
            }
        } catch (AmazonClientException e) {
            throw wrap(e, key);
        } finally {
            try {
                withVoidRetry(c -> c.deleteObject(bucketName, tempKey));
            } catch (AmazonClientException e) {
                LOG.warn("Error cleaning up temporary file", e);
            }
        }
    }

    @Override
    public void setFileLength(long length, String name) throws OXException {
        /*
         * Get the current length of the object
         */
        long contentLength = getContentLength(addPrefix(name));
        if (contentLength == length) {
            /* the file already has the desired length */
            return;
        }

        /**
         * Set the new file length: CopyPart/UploadPartCopy is preferred, because it does not require
         * to download the S3 object. But: UploadPartCopy might not be available/possible in all situations
         */
        if (isCopyPartAvailable(contentLength)) {
            setFileLengthWithCopyPart(length, name);
        } else {
            setFileLengthWithSpooling(length, name);
        }
    }

    /**
     * Creates a new arbitrary key (an unformatted string representation of a new random UUID), optionally prepended with the configured
     * prefix and delimiter.
     *
     * @param withPrefix <code>true</code> to prepend the prefix, <code>false</code>, otherwise
     *
     * @return A new UID string, optionally with prefix and delimiter, e.g. <code>[prefix]/067e61623b6f4ae2a1712470b63dff00</code>.
     */
    private String generateKey(boolean withPrefix) {
        String uuid = UUIDs.getUnformattedString(UUID.randomUUID());
        return withPrefix ? new StringBuilder(prefix).append(DELIMITER).append(uuid).toString() : uuid;
    }

    /**
     * Prepends the configured prefix and delimiter character sequence to the supplied name.
     *
     * @param name The name to prepend the prefix
     * @return The name with prefix
     */
    private String addPrefix(String name) {
        return addPrefix0(name, null);
    }

    /**
     * Prepends the configured prefix and delimiter character sequence to the supplied name.
     *
     * @param name The name to prepend the prefix
     * @param optBuilder The string builder to use or <code>null</code>
     * @return The name with prefix
     */
    private String addPrefix0(String name, StringBuilder optBuilder) {
        StringBuilder sb = optBuilder;
        if (sb == null) {
            sb = new StringBuilder(64);
        } else {
            if (sb.length() > 0) {
                sb.setLength(0);
            }
        }
        return sb.append(prefix).append(DELIMITER).append(name).toString();
    }

    /**
     * Prepends the configured prefix and delimiter character sequence to the supplied names.
     *
     * @param names The names to prepend the prefix
     * @return The names with prefix in an array
     */
    private String[] addPrefix(Collection<String> names) {
        if (names == null || names.isEmpty()) {
            return Strings.getEmptyStrings();
        }

        String[] keys = new String[names.size()];
        StringBuilder sb = new StringBuilder(64);
        int i = 0;
        for (String name : names) {
            keys[i++] = addPrefix0(name, sb);
        }
        return keys;
    }

    /**
     * Prepends the configured prefix and delimiter character sequence to the supplied names.
     *
     * @param names The names to prepend the prefix
     * @return The names with prefix in an array
     */
    private String[] addPrefix(String[] names) {
        if (names == null || names.length <= 0) {
            return Strings.getEmptyStrings();
        }

        String[] keys = new String[names.length];
        StringBuilder sb = new StringBuilder(64);
        for (int i = names.length; i-- > 0;) {
            keys[i] = addPrefix0(names[i], sb);
        }
        return keys;
    }

    /**
     * Strips the prefix and delimiter character sequence to the supplied key.
     *
     * @param key The key to strip the prefix from
     * @return The key without prefix
     */
    private String removePrefix(String key) {
        int idx = prefix.length() + 1; // Prefix length plus delimiter character
        if (idx > key.length() || false == key.startsWith(new StringBuilder(prefix).append(DELIMITER).toString())) {
            throw new IllegalArgumentException(key);
        }
        return key.substring(idx);
    }

    /**
     * Puts a single upload chunk to amazon s3.
     *
     * @param key The object key
     * @param chunk The chunk to store
     * @return The put object result passed from the client
     */
    private PutObjectResult uploadSingle(String key, S3UploadChunk chunk) throws OXException {
        ObjectMetadata metadata = prepareMetadataForSSE(new ObjectMetadata());
        if (client.getEncryptionConfig().isClientEncryptionEnabled()) {
            metadata.addUserMetadata(UNENCRYPTED_CONTENT_LENGTH, Long.toString(chunk.getSize()));
        } else {
            metadata.setContentLength(chunk.getSize());
            metadata.setContentMD5(chunk.getMD5Digest());
        }

        PutObjectResult result;
        InputStream data = null;
        try {
            PutObjectRequest request;
            Optional<File> optTempFile = chunk.getTempFile();
            if (optTempFile.isPresent()) {
                request = new PutObjectRequest(bucketName, key, optTempFile.get());
                request.setMetadata(metadata);
            } else {
                // It is a ByteArrayInputStream if temporary file is absent
                data = chunk.getData();
                request = new PutObjectRequest(bucketName, key, data, metadata);
                request.getRequestClientOptions().setReadLimit((int) chunk.getSize() + 1);
            }
            result = withRetry(c -> c.putObject(request));
        } finally {
            Streams.close(data);
        }
        applyUnencryptedContentLengthIfNeeded(key, chunk.getSize());
        return result;
    }

    /**
     * Uploads a single part of a multipart upload.
     *
     * @param key The key to store
     * @param uploadID the upload ID
     * @param partNumber The part number of the chunk
     * @param chunk the chunk to store
     * @param lastPart <code>true</code> if this is the last part, <code>false</code>, otherwise
     * @return The put object result passed from the client
     */
    private UploadPartResult uploadPart(String key, String uploadID, int partNumber, S3UploadChunk chunk, boolean lastPart) throws OXException {
        InputStream data = null;
        try {
            UploadPartRequest request;
            Optional<File> optTempFile = chunk.getTempFile();
            if (optTempFile.isPresent()) {
                File tempFile = optTempFile.get();
                request = new UploadPartRequest()
                    .withBucketName(bucketName)
                    .withKey(key)
                    .withUploadId(uploadID)
                    .withFile(tempFile)
                    .withPartSize(tempFile.length())
                    .withPartNumber(partNumber)
                    .withLastPart(lastPart);
            } else {
                // It is a ByteArrayInputStream if temporary file is absent
                data = chunk.getData();
                request = new UploadPartRequest()
                    .withBucketName(bucketName)
                    .withKey(key)
                    .withUploadId(uploadID)
                    .withInputStream(data)
                    .withPartSize(chunk.getSize())
                    .withPartNumber(partNumber)
                    .withLastPart(lastPart);
                request.getRequestClientOptions().setReadLimit((int) chunk.getSize() + 1);
            }
            String md5Digest = chunk.getMD5Digest();
            if (null != md5Digest) {
                request.withMD5Digest(md5Digest);
            }
            return withRetry(c -> c.uploadPart(request));
        } finally {
            Streams.close(data, chunk);
        }
    }

    /**
     * Appends data to an existing S3 object
     * <p>
     * This method gets the object stream from the existing S3 object, concatenates it with the new data, and stores it as new file on
     * the S3. Afterwards the original object is replaced with the new one.
     * </p>
     *
     * @param file The data to append as {@link InputStream}
     * @param name The name of the object to append the data to
     * @return The new size of the modified object
     */
    private long appendToFileWithSpooling(InputStream file, String name) throws OXException {
        /*
         * get existing object
         */
        String tempName = null;
        String key = addPrefix(name);
        S3Object s3Object = null;
        SequenceInputStream inputStream = null;
        try {
            s3Object = getObject(key);
            /*
             * append both streams at temporary location
             */
            inputStream = new SequenceInputStream(s3Object.getObjectContent(), file);
            tempName = saveNewFile(inputStream, false);
        } finally {
            Streams.close(inputStream, s3Object);
        }
        /*
         * replace old file, cleanup
         */
        String tempKey = addPrefix(tempName);
        try {
            ObjectMetadata newMetadata = cloneMetadata(getMetadata(tempKey));
            CopyObjectRequest copyObjectRequest = new CopyObjectRequest(bucketName, tempKey, bucketName, key).withNewObjectMetadata(newMetadata);
            withRetry(c -> c.copyObject(copyObjectRequest));
            return getContentLength(key);
        } catch (AmazonClientException e) {
            throw wrap(e, key);
        } finally {
            try {
                withVoidRetry(c -> c.deleteObject(bucketName, tempKey));
            } catch (AmazonClientException e) {
                LOG.warn("Error cleaning up temporary file", e);
            }
        }
    }

    /**
     * Appends data to an existing S3 object.
     * <p>
     * This method uses a <i>UploadPartCopy</i> ({@link CopyPartRequest}) within a multipart upload request in order to create a new temporary S3 object with the appended data.
     * Afterwards the temporary object replaces the original.
     * </p>
     * <p>
     * Due to S3 restrictions, a UploadPartCopy is only possible if the source file, where to append the data, is larger than 5 MB.
     * </p>
     *
     * @param file The data to append as {@link InputStream}
     * @param name The name of the object to append the data to
     * @return The new size of the modified object
     * @throws OXException If appending data fails
     */
    private long appendToFileWithCopyPart(InputStream file, String name) throws OXException {
        S3ChunkedUpload chunkedUpload = null;
        String key = addPrefix(name);
        try {
            String tmpKey = generateKey(true);

            // Initialize the multipart upload
            String uploadID = withRetry(c -> c.initiateMultipartUpload(new InitiateMultipartUploadRequest(bucketName, tmpKey).withObjectMetadata(prepareMetadataForSSE(new ObjectMetadata()))).getUploadId());

            boolean uploadCompleted = false;
            try {
                int partNumber = 1;

                // UploadPartCopy: Copy from the existing object into an S3 tmp Object
                //@formatter:off
                CopyPartRequest copyPartRequest = new CopyPartRequest()
                    .withUploadId(uploadID)
                    .withPartNumber(partNumber++)
                    .withSourceBucketName(bucketName)
                    .withSourceKey(key)
                    .withDestinationBucketName(bucketName)
                    .withDestinationKey(tmpKey);
                //@formatter:on
                CopyPartResult copyPartResult = withRetry(c -> c.copyPart(copyPartRequest));
                if (copyPartResult == null) {
                    /* cannot perform partial copy because of some not satisfied constrains; fallback to spooling */
                    return appendToFileWithSpooling(file, name);
                }
                List<PartETag> parts = new ArrayList<PartETag>();
                parts.add(copyPartResult.getPartETag());

                // Append the new data to the tmp object in chunks
                Thread currentThread = Thread.currentThread();
                chunkedUpload = new S3ChunkedUpload(file, client.getEncryptionConfig().isClientEncryptionEnabled(), client.getChunkSize());
                while (chunkedUpload.hasNext()) {
                    try (S3UploadChunk chunk = chunkedUpload.next()) {
                        boolean lastChunk = chunkedUpload.hasNext() == false;
                        parts.add(uploadPart(tmpKey, uploadID, partNumber++, chunk, lastChunk).getPartETag());
                        if (currentThread.isInterrupted()) {
                            throw OXException.general("Upload to S3 aborted");
                        }
                    }
                }

                // Finish the multipart upload
                withRetry(c -> c.completeMultipartUpload(new CompleteMultipartUploadRequest(bucketName, tmpKey, uploadID, parts)));
                uploadCompleted = true;

                // Replace the old file , cleanup
                try {
                    CopyObjectRequest copyObjectRequest = new CopyObjectRequest(bucketName, tmpKey, bucketName, key);
                    ObjectMetadata metadata = prepareMetadataForSSE(new ObjectMetadata());
                    copyObjectRequest.setNewObjectMetadata(metadata);
                    withRetry(c -> c.copyObject(copyObjectRequest));
                    withVoidRetry(c -> c.deleteObject(bucketName, tmpKey));
                    return getContentLength(key);
                } catch (AmazonClientException e) {
                    throw wrap(e, key);
                }
            } catch (IOException e) {
                throw FileStorageCodes.IOERROR.create(e, e.getMessage());
            } finally {
                if (!uploadCompleted) {
                    try {
                        withVoidRetry(c -> c.abortMultipartUpload(new AbortMultipartUploadRequest(bucketName, tmpKey, uploadID)));
                    } catch (Exception e) {
                        LOG.warn("Error aborting multipart upload", e);
                    }
                }
            }
        } finally {
            Streams.close(chunkedUpload);
        }
    }
}
