/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.drive.client.windows.servlet;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.ajax.SessionServlet;
import com.openexchange.ajax.helper.BrowserDetector;
import com.openexchange.drive.client.windows.service.BrandingService;
import com.openexchange.drive.client.windows.service.DriveUpdateService;
import com.openexchange.drive.client.windows.service.internal.Utils;
import com.openexchange.exception.OXException;
import com.openexchange.java.Streams;
import com.openexchange.tools.encoding.Helper;
import com.openexchange.tools.servlet.http.Tools;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tools.session.ServerSessionAdapter;

/**
 * {@link InstallServlet} is a servlet to initially download the drive client
 *
 * @author <a href="mailto:kevin.ruthmann@open-xchange.com">Kevin Ruthmann</a>
 * @since v7.8.1
 */
public class InstallServlet extends SessionServlet {

    private static final long serialVersionUID = 6099648933040174331L;

    private static final Logger LOG = LoggerFactory.getLogger(InstallServlet.class);

    private final DriveUpdateService updateService;

    public InstallServlet(DriveUpdateService updateService) {
        super();
        this.updateService = updateService;
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        try {
            ServerSession session = getSessionObject(req);
            if (null == session) {
                resp.sendError(HttpServletResponse.SC_FORBIDDEN);
                return;
            }

            String branding = BrandingService.getBranding(session);
            String fileName = updateService.getExeFileName(branding);
            if (!Utils.hasPermissions(ServerSessionAdapter.valueOf(session).getUserConfiguration(), updateService.getNecessaryPermission())) {
                resp.sendError(HttpServletResponse.SC_FORBIDDEN);
                return;
            }

            if (fileName == null) {
                LOG.warn("No file found for brand {}. Please review your configuration.", branding);
                resp.sendError(HttpServletResponse.SC_NOT_FOUND);
                return;
            }

            BrowserDetector detector = BrowserDetector.detectorFor(req.getHeader(Tools.HEADER_AGENT));
            resp.setStatus(HttpServletResponse.SC_OK);
            resp.setContentType("application/octet-stream");
            resp.setHeader("Content-disposition", "attachment; filename=\"" + Helper.escape(Helper.encodeFilename(fileName, "UTF-8", detector.isMSIE())) + "\"");
            Tools.removeCachingHeader(resp);

            InputStream is = updateService.getFile(fileName, branding);
            try {
                OutputStream out = resp.getOutputStream();
                byte[] buf = new byte[8192];
                for (int length; (length = is.read(buf)) > 0;) {
                    out.write(buf, 0, length);
                }
                out.flush();
            } finally {
                Streams.close(is);
            }
        } catch (OXException e) {
            LOG.error("", e);
            resp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }
    }

}
