/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2017 Open-Xchange GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.data;

import static com.openexchange.build.utils.Definitions.DEBIAN_CHANGELOG_PATTERN;
import static com.openexchange.build.utils.Definitions.DEBIAN_DF;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.regex.Matcher;
import com.openexchange.build.utils.Definitions;
import com.openexchange.build.utils.Version;

/**
 * {@link DebianChangeLogEntry}
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class DebianChangeLogEntry extends SpecChangeLogEntry {

    private final String packageName;
    private final Version version;
    private final String release;

    public DebianChangeLogEntry(String packageName, String version, String release, String comment, String author, String date) throws Exception {
        this(packageName, version, release, comment, author, parseDate(date));
    }
    public DebianChangeLogEntry(String packageName, String version, String release, String comment, String author, Date date) throws Exception {
        super(date, author, comment);
        this.packageName = packageName;
        this.version = Version.parse(version);
        this.release = release;
    }

    private static Date parseDate(String date) throws Exception {
        try {
            return new SimpleDateFormat(DEBIAN_DF, Locale.ENGLISH).parse(date);
        } catch (ParseException e) {
            throw new Exception("Unable to parse date for changelog entry: \"" + date + "\"", e);
        }
    }

    public static DebianChangeLogEntry parse(String entry) throws Exception {
        Matcher matcher = DEBIAN_CHANGELOG_PATTERN.matcher(entry);
        if (matcher.find()) {
            return new DebianChangeLogEntry(matcher.group(1), matcher.group(2), matcher.group(8), matcher.group(9), matcher.group(10), matcher.group(11));
        }
        throw new Exception("Can not parse debian changelog entry:\n Start without quote -> \"" + entry + "\" <- End without quote");
    }

    public int getMajor() {
        return version.getMajor();
    }

    public int getMinor() {
        return version.getMinor();
    }

    public int getPatch() {
        return version.getPatch();
    }

    public String getRevision() {
        return version.getRevision();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(packageName);
        sb.append(" (");
        sb.append(version.toString());
        sb.append(") ");
        sb.append(release);
        sb.append("\n\n  * ");
        sb.append(getComment());
        sb.append("\n\n -- ");
        sb.append(getAuthor());
        sb.append("  ");
        DateFormat df = new SimpleDateFormat(Definitions.DEBIAN_DF, Locale.ENGLISH);
        df.getCalendar().setTime(getDate());
        sb.append(df.format(getDate()));
        sb.append('\n');
        return sb.toString();
    }
}
