/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mail.authenticity.mechanism.spf;

import com.openexchange.mail.authenticity.mechanism.AuthenticityMechanismResult;

/**
 * {@link SPFResult} - Specifies the possible SPF results.
 * The ordinal defines the significance of each result.
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 * @see <a href="https://tools.ietf.org/html/rfc7208#section-8">RFC-7208, Section 8</a>
 */
public enum SPFResult implements AuthenticityMechanismResult {

    /**
     * The client is authorised to inject mail with the given identity. The domain
     * can now, in the sense of reputation, be considered responsible for sending
     * the message. Further policy checks can now proceed with confidence in the
     * legitimate use of the identity.
     *
     * @see <a href="https://tools.ietf.org/html/rfc7208#section-8.3">RFC-7208, Section 8.3</a>
     */
    PASS("Pass", "pass"),
    /**
     * <p>Indicates that although a policy for the identity was discovered, there is
     * no definite assertion (positive or negative) about the client.</p>
     *
     * <p>A "neutral" result MUST be treated exactly like the "none" result;
     * the distinction exists only for informational purposes. Treating
     * "neutral" more harshly than "none" would discourage ADMDs from
     * testing the use of SPF records.</p>
     *
     * @see <a href="https://tools.ietf.org/html/rfc7208#section-8.2">RFC-7208, Section 8.2</a>
     */
    NEUTRAL("Neutral", "neutral"),
    /**
     * Indicates that some local policy mechanism was applied
     * that augments or even replaces (i.e., overrides) the result returned
     * by the authentication mechanism.
     *
     * @see <a href="https://tools.ietf.org/html/rfc7601#section-2.4">RFC-7601, Section 2.4</a>
     */
    POLICY("Policy", "policy"),
    /**
     * The SPF verifier has no information at all about the authorisation
     * or lack thereof of the client to use the checked identity or identities.
     * The check_host() function completed without errors but was not able to
     * reach any conclusion.
     *
     * @see <a href="https://tools.ietf.org/html/rfc7208#section-8.1">RFC-7208, Section 8.1</a>
     */
    NONE("None", "none"),
    /**
     * The SPF verifier encountered a transient (generally DNS) error while performing the check.
     * Checking software can choose to accept or temporarily reject the message. If the message
     * is rejected during the SMTP transaction for this reason, the software SHOULD use an SMTP
     * reply code of 451 and, if supported, the 4.4.3 enhanced status code (see
     * <a href="https://tools.ietf.org/html/rfc3463#section-3.5">Section 3.5 of [RFC3463]</a>).
     * These errors can be caused by problems in either the sender's or receiver's DNS software.
     *
     * @see <a href="https://tools.ietf.org/html/rfc7208#section-8.6">RFC-7208, Section 8.6</a>
     */
    TEMPERROR("Temporary Error", "temperror"),
    /**
     * The domain's published records could not be correctly interpreted. This signals an error
     * condition that definitely requires DNS operator intervention to be resolved. If the message
     * is rejected during the SMTP transaction for this reason, the software SHOULD use an SMTP
     * reply code of 550 and, if supported, the 5.5.2 enhanced status code (see
     * <a href="https://tools.ietf.org/html/rfc3463#section-3.6">[RFC3463], Section 3.6</a>).
     * Be aware that if the ADMD uses macros (<a href="https://tools.ietf.org/html/rfc7208#section-7">Section 7</a>),
     * it is possible that this result is due to the checked identities having an unexpected format.
     * It is also possible that this result is generated by certain SPF verifiers due to the input arguments
     * having an unexpected format.
     *
     * @see <a href="https://tools.ietf.org/html/rfc7208#section-8.7">RFC-7208, Section 8.7</a>
     */
    PERMERROR("Permanent Error", "permerror"),
    /**
     * Ought to be treated as somewhere between "fail" and "neutral"/"none". The ADMD
     * believes the host is not authorised but is not willing to make a strong policy
     * statement. Receiving software SHOULD NOT reject the message based solely on this
     * result, but MAY subject the message to closer scrutiny than normal.
     *
     * @see <a href="https://tools.ietf.org/html/rfc7208#section-8.5">RFC-7208, Section 8.5</a>
     */
    SOFTFAIL("Soft Fail", "softfail"),
    /**
     * An explicit statement that the client is not authorised to use the domain in the
     * given identity. Disposition of SPF fail messages is a matter of local policy.
     *
     * @see <a href="https://tools.ietf.org/html/rfc7208#section-8.4">RFC-7208, Section 8.4</a>
     */
    FAIL("Fail", "fail"),
    ;

    private final String displayName;
    private final String technicalName;

    /**
     * Initialises a new {@link SPFResult}.
     */
    private SPFResult(String displayName, String technicalName) {
        this.displayName = displayName;
        this.technicalName = technicalName;
    }

    @Override
    public String getDisplayName() {
        return displayName;
    }

    @Override
    public String getTechnicalName() {
        return technicalName;
    }

    @Override
    public int getCode() {
        return ordinal();
    }
}
