/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.groupware.update.internal;

import com.openexchange.groupware.update.UpdateTaskV2;

/**
 * Checks if all dependencies are resolved or will be resolved before the current update task is executed.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public class DependenciesResolvedChecker implements DependencyChecker {

    public DependenciesResolvedChecker() {
        super();
    }

    @Override
    public boolean check(UpdateTaskV2 task, String[] executed, UpdateTaskV2[] enqueued, UpdateTaskV2[] toExecute) {
        // Check all dependencies.
        for (String dependency : task.getDependencies()) {
            if (!dependencyFulfilled(dependency, executed, enqueued)) {
                return false;
            }
        }
        return true;
    }

    boolean dependencyFulfilled(String dependency, String[] executed, UpdateTaskV2[] enqueued) {
        for (String taskName : executed) {
            if (taskName.equals(dependency)) {
                return true;
            }
        }
        for (UpdateTaskV2 task : enqueued) {
            if (task.getClass().getName().equals(dependency)) {
                return true;
            }
        }
        return false;
    }
}
