/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.security.manager.configurationReader;

import java.io.File;
import java.util.LinkedList;
import java.util.List;
import com.openexchange.config.ConfigurationService;

/**
 * {@link ConfigurationFileParser} Loads all of the configurations in the security-manager.list that
 * will require directory access
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v7.10.3
 */
public class ConfigurationFileParser {

    private static final String SECURITY_FILE_SUFFIX = ".list";
    private static final String SECURITY_FOLDER = "security";
    private final ConfigurationService configService;

    public ConfigurationFileParser(ConfigurationService configService) {
        this.configService = configService;
    }

    /**
     * Read the data from a file, add each valid line to list
     *
     * @param file File to read
     * @param list List to store valid lines
     */
    private void parseFile(File file, List<String> list) {
        String data = configService.getText(file.getName());
        if (data == null) {
            return;
        }
        String[] lines = data.split("\n");
        for (String line : lines) {
            line = line.trim();
            if (line.indexOf("#") != 0 && !line.isEmpty()) {
                list.add(line);
            }
        }
    }

    /**
     * Read through the security directory and return List of configuration options
     *
     * @return List of configurations found in the files
     */
    public List<String> getConfigList() {
        File folder = configService.getDirectory(SECURITY_FOLDER);
        List<String> list = new LinkedList<>();
        if (null != folder && folder.exists() && folder.isDirectory()) {
            File[] files = folder.listFiles();
            if (null == files) {
                return list;
            }
            for (File file : files) {
                if (file.getName().endsWith(SECURITY_FILE_SUFFIX)) {
                    parseFile(file, list);
                }
            }
        }
        return list;
    }

}
