/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.passwordchange.servlet;

import com.openexchange.exception.Category;
import com.openexchange.exception.DisplayableOXExceptionCode;
import com.openexchange.exception.OXException;
import com.openexchange.exception.OXExceptionFactory;
import com.openexchange.exception.OXExceptionStrings;

public enum PasswordChangeServletExceptionCode implements DisplayableOXExceptionCode {

    /**
     * A JSON error occurred: %1$s
     */
    JSON_ERROR("A JSON error occurred: %1$s", CATEGORY_ERROR, PasswordChangeServletExceptionMessage.PW_CHANGE_ERROR_MSG),
    /**
     * Invalid locale string: %1$s
     */
    INVALID_LOCALE_STR("Invalid locale string: %1$s", CATEGORY_ERROR, PasswordChangeServletExceptionMessage.PW_CHANGE_ERROR_MSG),
    /**
     * Unsupported action %1$s via %2$s request
     */
    UNSUPPORTED_ACTION("Unsupported action %1$s via %2$s request", CATEGORY_ERROR, PasswordChangeServletExceptionMessage.PW_CHANGE_ERROR_MSG),
    /**
     * Missing parameter %1$s
     */
    MISSING_PARAM("Missing parameter %1$s", CATEGORY_ERROR, PasswordChangeServletExceptionMessage.PW_CHANGE_ERROR_MSG),
    /**
     * Password changed successfully. Please logout and login back again.
     */
    PW_CHANGE_SUCCEEDED("Password changed successfully. Please logout and login back again.", CATEGORY_WARNING,
        PasswordChangeServletExceptionMessage.PW_CHANGE_SUCCEEDED_MSG),

    ;

    private final String message;
    private final int detailNumber;
    private final Category category;
    private final String displayMessage;

    private PasswordChangeServletExceptionCode(final String message, final Category category, String displayMessage) {
        this.message = message;
        this.detailNumber = ordinal() + 1;
        this.category = category;
        this.displayMessage = displayMessage != null ? displayMessage : OXExceptionStrings.MESSAGE;
    }

    @Override
    public String getPrefix() {
        return "PSW";
    }

    @Override
    public Category getCategory() {
        return category;
    }

    @Override
    public int getNumber() {
        return detailNumber;
    }

    @Override
    public String getMessage() {
        return message;
    }
    
    @Override
    public String getDisplayMessage() {
        return displayMessage;
    }

    @Override
    public boolean equals(final OXException e) {
        return OXExceptionFactory.getInstance().equals(this, e);
    }

    /**
     * Creates a new {@link OXException} instance pre-filled with this code's attributes.
     *
     * @return The newly created {@link OXException} instance
     */
    public OXException create() {
        return OXExceptionFactory.getInstance().create(this, new Object[0]);
    }

    /**
     * Creates a new {@link OXException} instance pre-filled with this code's attributes.
     *
     * @param args The message arguments in case of printf-style message
     * @return The newly created {@link OXException} instance
     */
    public OXException create(final Object... args) {
        return OXExceptionFactory.getInstance().create(this, (Throwable) null, args);
    }

    /**
     * Creates a new {@link OXException} instance pre-filled with this code's attributes.
     *
     * @param cause The optional initial cause
     * @param args The message arguments in case of printf-style message
     * @return The newly created {@link OXException} instance
     */
    public OXException create(final Throwable cause, final Object... args) {
        return OXExceptionFactory.getInstance().create(this, cause, args);
    }
}
