/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mail.compose.impl.cleanup;

import static com.openexchange.java.Autoboxing.I;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import com.openexchange.config.ConfigTools;
import com.openexchange.config.cascade.ConfigView;
import com.openexchange.config.cascade.ConfigViewFactory;
import com.openexchange.config.cascade.ConfigViews;
import com.openexchange.exception.OXException;
import com.openexchange.mail.compose.CompositionSpaceServiceFactory;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.timer.TimerService;

/**
 * {@link CompositionSpaceCleanUpScheduler}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.10.3
 */
public class CompositionSpaceCleanUpScheduler {

    /** The logger constant */
    static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(CompositionSpaceCleanUpScheduler.class);

    private static final AtomicReference<CompositionSpaceCleanUpScheduler> INSTANCE_REFERENCE = new AtomicReference<>(null);

    /**
     * Initializes the instance
     *
     * @param compositionSpaceServiceFactory The service factory to use
     * @param services The service look-up to use
     * @return The freshly initialized instance or empty if already initialized before
     */
    public static synchronized Optional<CompositionSpaceCleanUpScheduler> initInstance(CompositionSpaceServiceFactory compositionSpaceServiceFactory, ServiceLookup services) {
        if (INSTANCE_REFERENCE.get() != null) {
            // Already initialized
            return Optional.empty();
        }

        CompositionSpaceCleanUpScheduler instance = new CompositionSpaceCleanUpScheduler(compositionSpaceServiceFactory, services);
        INSTANCE_REFERENCE.set(instance);
        return Optional.of(instance);
    }

    /**
     * Releases the instance
     */
    public static synchronized void releaseInstance() {
        INSTANCE_REFERENCE.getAndSet(null);
    }

    /**
     * Gets the instance.
     *
     * @return The instance
     */
    public static CompositionSpaceCleanUpScheduler getInstance() {
        return INSTANCE_REFERENCE.get();
    }

    // -------------------------------------------------------------------------------------------------------------------------------------

    private final CompositionSpaceServiceFactory compositionSpaceServiceFactory;
    private final ServiceLookup services;


    /**
     * Initializes a new {@link CompositionSpaceCleanUpScheduler}.
     *
     * @param compositionSpaceServiceFactory The service factory to use
     * @param services The service look-up to use
     */
    private CompositionSpaceCleanUpScheduler(CompositionSpaceServiceFactory compositionSpaceServiceFactory, ServiceLookup services) {
        super();
        this.compositionSpaceServiceFactory = compositionSpaceServiceFactory;
        this.services = services;
    }

    /**
     * Schedules clean-up task for given arguments.
     *
     * @param session The session
     * @return <code>true</code> if caller scheduled clean-up task; otherwise <code>false</code>
     */
    public boolean scheduleCleanUpFor(Session session) {
        TimerService timerService = services.getOptionalService(TimerService.class);
        if (timerService == null) {
            return false;
        }

        timerService.schedule(new CleanUpTask(session, compositionSpaceServiceFactory, services), 5000L);
        LOG.debug("Scheduled composition space clean-up task for user {} in context {}", I(session.getUserId()), I(session.getContextId()));
        return true;
    }

    // -------------------------------------------------------------------------------------------------------------------------------------

    private static class CleanUpTask implements Runnable {

        private final Session session;
        private final CompositionSpaceServiceFactory compositionSpaceServiceFactory;
        private final ServiceLookup services;

        /**
         * Initializes a new {@link CleanUpTask}.
         *
         * @param session The session for which this task is started
         * @param compositionSpaceServiceFactory The service factory used to drop expired composition spaces
         * @param services The service look-up
         */
        CleanUpTask(Session session, CompositionSpaceServiceFactory compositionSpaceServiceFactory, ServiceLookup services) {
            super();
            this.session = session;
            this.compositionSpaceServiceFactory = compositionSpaceServiceFactory;
            this.services = services;
        }

        @Override
        public void run() {
            try {
                long maxIdleTimeMillis = getMaxIdleTimeMillis(session);
                if (maxIdleTimeMillis > 0) {
                    compositionSpaceServiceFactory.createServiceFor(session).closeExpiredCompositionSpaces(maxIdleTimeMillis);
                }
            } catch (Exception e) {
                LOG.error("Failed to clean-up expired composition spaces for user {} in context {}", I(session.getUserId()), I(session.getContextId()), e);
            }
        }

        private long getMaxIdleTimeMillis(Session session) throws OXException {
            String defaultValue = "1W";

            ConfigViewFactory viewFactory = services.getOptionalService(ConfigViewFactory.class);
            if (null == viewFactory) {
                return ConfigTools.parseTimespan(defaultValue);
            }

            ConfigView view = viewFactory.getView(session.getUserId(), session.getContextId());
            return ConfigTools.parseTimespan(ConfigViews.getDefinedStringPropertyFrom("com.openexchange.mail.compose.maxIdleTimeMillis", defaultValue, view));
        }
    } // End of class CleanUpTask

}
