/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.imagetransformation.java.transformations;

import static com.openexchange.imagetransformation.java.transformations.Utils.getFileStream;
import static com.openexchange.imagetransformation.java.transformations.Utils.getImageInputStream;
import static com.openexchange.imagetransformation.java.transformations.Utils.getImageReader;
import static com.openexchange.imagetransformation.java.transformations.Utils.getRequiredResolution;
import static com.openexchange.imagetransformation.java.transformations.Utils.readExifOrientation;
import static com.openexchange.imagetransformation.java.transformations.Utils.removeTransparencyIfNeeded;
import static com.openexchange.imagetransformation.java.transformations.Utils.selectImage;
import static com.openexchange.java.Autoboxing.I;
import static com.openexchange.tools.images.ImageTransformationUtility.canRead;
import static com.openexchange.tools.images.ImageTransformationUtility.getImageFormat;
import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.ImageOutputStream;
import com.openexchange.ajax.container.ThresholdFileHolder;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.exception.OXException;
import com.openexchange.imagetransformation.BasicTransformedImage;
import com.openexchange.imagetransformation.Constants;
import com.openexchange.imagetransformation.ImageInformation;
import com.openexchange.imagetransformation.ImageTransformationDeniedIOException;
import com.openexchange.imagetransformation.ImageTransformationSignaler;
import com.openexchange.imagetransformation.ImageTransformations;
import com.openexchange.imagetransformation.ScaleType;
import com.openexchange.imagetransformation.TransformationContext;
import com.openexchange.imagetransformation.TransformedImage;
import com.openexchange.imagetransformation.Utility;
import com.openexchange.java.Streams;
import com.openexchange.tools.images.DefaultTransformedImageCreator;
import com.openexchange.tools.stream.CountingInputStream;
import com.openexchange.tools.stream.CountingInputStream.IOExceptionCreator;
import com.openexchange.tools.stream.UnsynchronizedByteArrayOutputStream;

/**
 * {@link ImageTransformationsImpl}
 *
 * Default {@link ImageTransformations} implementation.
 *
 * @author <a href="mailto:martin.herfurth@open-xchange.com">Martin Herfurth</a>
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 */
public class ImageTransformationsImpl implements ImageTransformations {

    private static org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(ImageTransformationsImpl.class);

    static int waitTimeoutSeconds() {
        return Utility.waitTimeoutSeconds();
    }

    static long maxSize() {
        return Utility.maxSize();
    }

    static long maxResolution() {
        return Utility.maxResolution();
    }

    static float preferThumbnailThreshold() {
        return Utility.preferThumbnailThreshold();
    }

    private static final IOExceptionCreator IMAGE_SIZE_EXCEEDED_EXCEPTION_CREATOR = new IOExceptionCreator() {

        @Override
        public IOException createIOException(long total, long max) {
            if (total > 0 && max > 0) {
                return new ImageTransformationDeniedIOException(new StringBuilder("Image transformation denied. Size is too big. (current=").append(total).append(", max=").append(max).append(')').toString());
            }
            return new ImageTransformationDeniedIOException("Image transformation denied. Size is too big.");
        }
    };

    private static IOException createResolutionExceededIOException(long maxResolution, int resolution) {
        return new ImageTransformationDeniedIOException(new StringBuilder("Image transformation denied. Resolution is too high. (current=").append(resolution).append(", max=").append(maxResolution).append(')').toString());
    }

    private static final Comparator<ImageTransformation> TRANSFORMATION_COMPARATOR = new Comparator<ImageTransformation>() {

        @Override
        public int compare(ImageTransformation t1, ImageTransformation t2) {
            boolean isRotate1 = t1 == RotateTransformation.getInstance();
            boolean isRotate2 = t2 == RotateTransformation.getInstance();
            return isRotate1 ? (isRotate2 ? 0 : -1) : (isRotate2 ? 1 : 0);
        }
    };

    // ------------------------------------------------------------------------------------------------------------------------------ //

    private final TransformationContext transformationContext;
    private final InputStream sourceImageStream;
    private final IFileHolder sourceImageFile;
    private final List<ImageTransformation> transformations;
    private BufferedImage sourceImage;
    private ImageInformation imageInformation;
    private boolean compress;
    protected final Object optSource;

    private ImageTransformationsImpl(BufferedImage sourceImage, InputStream sourceImageStream, IFileHolder imageFile, Object optSource) {
        super();
        this.optSource = optSource;

        if (null == imageFile) {
            this.sourceImageStream = sourceImageStream;
            this.sourceImageFile = null;
        } else {
            if (imageFile.repetitive()) {
                this.sourceImageStream = null;
                this.sourceImageFile = imageFile;
            } else {
                try {
                    this.sourceImageStream = imageFile.getStream();
                    this.sourceImageFile = null;
                } catch (OXException e) {
                    throw new IllegalStateException(e.getMessage(), e);
                }
            }
        }

        this.sourceImage = sourceImage;
        this.transformations = new ArrayList<>();
        this.transformationContext = new TransformationContext();
    }

    /**
     * Adds specified transformation to the chain and re-orders that chain to ensure auto-rotation is at chain's head (as previously executed transformations loose EXIF data).
     *
     * @param transformation The transformation to add
     */
    private void addAndSort(ImageTransformation transformation) {
        transformations.add(transformation);
        if (transformations.size() > 1) {
            Collections.sort(transformations, TRANSFORMATION_COMPARATOR);
        }
    }

    /**
     * Initializes a new {@link ImageTransformationsImpl}.
     *
     * @param sourceImage The source image
     * @param optSource The source for this invocation; if <code>null</code> calling {@link Thread} is referenced as source
     */
    public ImageTransformationsImpl(final BufferedImage sourceImage, final Object optSource) {
        this(sourceImage, null, null, optSource);
    }

    /**
     * Initializes a new {@link ImageTransformationsImpl}.
     *
     * @param sourceImage The source image
     * @param optSource The source for this invocation; if <code>null</code> calling {@link Thread} is referenced as source
     */
    public ImageTransformationsImpl(final InputStream sourceImageStream, final Object optSource) {
        this(null, sourceImageStream, null, optSource);
    }

    /**
     * Initializes a new {@link ImageTransformationsTask}.
     *
     * @param imageFile The image file
     * @param optSource The source for this invocation; if <code>null</code> calling {@link Thread} is referenced as source
     */
    public ImageTransformationsImpl(IFileHolder imageFile, Object optSource) {
        this(null, null, imageFile, optSource);
    }

    @Override
    public ImageTransformations rotate() {
        // No need to sort
        transformations.add(0, RotateTransformation.getInstance());
        return this;
    }

    @Override
    public ImageTransformations scale(int maxWidth, int maxHeight, ScaleType scaleType) {
        return scale(maxWidth, maxHeight, scaleType, false);
    }

    @Override
    public ImageTransformations scale(int maxWidth, int maxHeight, ScaleType scaleType, boolean shrinkOnly) {
        if (maxWidth > Constants.getMaxWidth()) {
            throw new IllegalArgumentException("Width " + maxWidth + " exceeds max. supported width " + Constants.getMaxWidth());
        }
        if (maxHeight > Constants.getMaxHeight()) {
            throw new IllegalArgumentException("Height " + maxHeight + " exceeds max. supported height " + Constants.getMaxHeight());
        }
        addAndSort(new ScaleTransformation(maxWidth, maxHeight, scaleType, shrinkOnly));
        return this;
    }

    @Override
    public ImageTransformations crop(int x, int y, int width, int height) {
        addAndSort(new CropTransformation(x, y, width, height));
        return this;
    }

    @Override
    public ImageTransformations compress() {
        this.compress = true;
        return this;
    }

    @Override
    public BufferedImage getImage() throws IOException {
        if (false == needsTransformation(null) && null != sourceImage) {
            return sourceImage;
        }
        // Get BufferedImage
        return getImage(null, null);
    }

    @Override
    public byte[] getBytes(String formatName) throws IOException {
        String imageFormat = getImageFormat(formatName);
        return innerGetBytes(imageFormat);
    }

    private byte[] innerGetBytes(final String imageFormat) throws IOException {
        return write(getImage(imageFormat, null), imageFormat);
    }

    @Override
    public InputStream getInputStream(String formatName) throws IOException {
        String imageFormat = getImageFormat(formatName);
        if (false == needsTransformation(imageFormat)) {
            // Nothing to do
            InputStream in = getFileStream(sourceImageFile);
            if (null != in) {
                return in;
            }
            in = sourceImageStream;
            if (null != in) {
                return in;
            }
        }
        // Perform transformations
        byte[] bytes = innerGetBytes(imageFormat);
        return null == bytes ? null : Streams.newByteArrayInputStream(bytes);
    }

    @Override
    public BasicTransformedImage getTransformedImage(String formatName) throws IOException {
        String imageFormat = getImageFormat(formatName);
        BufferedImage bufferedImage = getImage(imageFormat, null);
        return writeTransformedImage(bufferedImage, imageFormat);
    }

    @Override
    public TransformedImage getFullTransformedImage(String formatName) throws IOException {
        String imageFormat = getImageFormat(formatName);
        BufferedImage bufferedImage = getImage(imageFormat, null);
        return writeTransformedImage(bufferedImage, imageFormat);
    }

    /**
     * Gets a value indicating whether the denoted format name leads to transformations or not.
     *
     * @param formatName The format name
     * @return <code>true</code>, if there are transformations for the target image format, <code>false</code>, otherwise
     */
    private boolean needsTransformation(String formatName) {
        if (false == canRead(formatName)) {
            return false;
        }
        for (ImageTransformation transformation : transformations) {
            if (transformation.supports(formatName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Gets the resulting image after applying all transformations.
     *
     * @param formatName The image format to use, or <code>null</code> if not relevant
     * @param signaler The optional signaler or <code>null</code>
     * @return The transformed image
     * @throws IOException if an I/O error occurs
     */
    protected BufferedImage getImage(String formatName, ImageTransformationSignaler signaler) throws IOException {
        BufferedImage image = getSourceImage(formatName, signaler);

        if (null != image && image.getHeight() > 3 && image.getWidth() > 3) {
            for (ImageTransformation transformation : transformations) {
                if (transformation.supports(formatName)) {
                    image = transformation.perform(image, transformationContext, imageInformation);
                }
            }
        }
        image = removeTransparencyIfNeeded(image, formatName);
        return image;
    }

    /**
     * Gets the source image, either from the supplied buffered image or the supplied stream, extracting image metadata as needed.
     *
     * @param formatName The format to use, e.g. "jpeg" or "tiff"
     * @param signaler The optional signaler or <code>null</code>
     * @return The source image
     * @throws IOException
     */
    private BufferedImage getSourceImage(String formatName, ImageTransformationSignaler signaler) throws IOException {
        if (null == sourceImage) {
            if (null != sourceImageStream) {
                long maxSize = maxSize();
                long maxResolution = maxResolution();
                sourceImage = needsMetadata(formatName, maxSize, maxResolution) ? readAndExtractMetadataFromStream(sourceImageStream, formatName, maxSize, maxResolution, signaler) : read(sourceImageStream, formatName, signaler);
            } else if (null != sourceImageFile) {
                long maxSize = maxSize();
                long maxResolution = maxResolution();
                sourceImage = needsMetadata(formatName, maxSize, maxResolution) ? readAndExtractMetadataFromFile(sourceImageFile, formatName, maxSize, maxResolution, signaler) : read(getFileStream(sourceImageFile), formatName, signaler);
            }
        }
        return sourceImage;
    }

    /**
     * Gets a value indicating whether additional metadata is required for one of the transformations or not.
     *
     * @param formatName The format to use, e.g. "jpeg" or "tiff"
     * @param maxSize The max. size for an image
     * @param maxResolution The max. resolution for an image
     * @return <code>true</code>, if metadata is needed, <code>false</code>, otherwise
     */
    private boolean needsMetadata(String formatName, long maxSize, long maxResolution) {
        if (maxSize > 0 || maxResolution > 0) {
            // Limitations specified, thus meta-data is needed
            return true;
        }
        if (null == formatName || 0 == formatName.length()) {
            return false;
        }
        for (ImageTransformation transformation : transformations) {
            if (transformation.supports(formatName) && transformation.needsImageInformation()) {
                return true;
            }
        }
        return false;
    }

    private TransformedImage writeTransformedImage(BufferedImage image, String formatName) throws IOException {
        return new DefaultTransformedImageCreator().writeTransformedImage(image, formatName, transformationContext, needsCompression(formatName));
    }


    /**
     * Writes out an image into a byte-array.
     *
     * @param image The image to write
     * @param formatName The format to use, e.g. "jpeg" or "tiff"
     * @return The image data
     * @throws IOException
     */
    private byte[] write(BufferedImage image, String formatName) throws IOException {
        if (null == image) {
            return null;
        }
        UnsynchronizedByteArrayOutputStream outputStream = null;
        try {
            outputStream = new UnsynchronizedByteArrayOutputStream(8192);
            if (needsCompression(formatName)) {
                writeCompressed(image, formatName, outputStream, transformationContext);
            } else {
                write(image, formatName, outputStream);
            }
            return outputStream.toByteArray();
        } finally {
            Streams.close(outputStream);
        }
    }

    private boolean needsCompression(String formatName) {
        return this.compress && null != formatName && "jpeg".equalsIgnoreCase(formatName) || "jpg".equalsIgnoreCase(formatName);
    }

    private static void write(BufferedImage image, String formatName, OutputStream output) throws IOException {
        if (false == ImageIO.write(image, formatName, output)) {
            throw new IOException("no appropriate writer found for " + formatName);
        }
    }

    private static void writeCompressed(BufferedImage image, String formatName, OutputStream output, TransformationContext transformationContext) throws IOException {
        ImageWriter writer = null;
        ImageOutputStream imageOutputStream = null;
        try {
            Iterator<ImageWriter> iter = ImageIO.getImageWritersByFormatName(formatName);
            if (null == iter || false == iter.hasNext()) {
                iter = ImageIO.getImageWritersByMIMEType(formatName);
                if (null == iter || false == iter.hasNext()) {
                    throw new IOException("No image writer for format " + formatName);
                }
            }
            writer = iter.next();
            ImageWriteParam iwp = writer.getDefaultWriteParam();
            adjustCompressionParams(iwp);
            imageOutputStream = ImageIO.createImageOutputStream(output);
            writer.setOutput(imageOutputStream);
            IIOImage iioImage = new IIOImage(image, null, null);
            writer.write(null, iioImage, iwp);
            transformationContext.addExpense(ImageTransformations.LOW_EXPENSE);
        } finally {
            if (null != writer) {
                writer.dispose();
            }
            if (null != imageOutputStream) {
                imageOutputStream.close();
            }
        }
    }

    /**
     * Tries to adjust the default settings on the supplied image write parameters to apply compression, ignoring any
     * {@link UnsupportedOperationException}s that may occur.
     *
     * @param parameters The parameters to adjust for compression
     */
    private static void adjustCompressionParams(ImageWriteParam parameters) {
        try {
            parameters.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
        } catch (UnsupportedOperationException e) {
            LOG.debug("", e);
        }
        try {
            parameters.setProgressiveMode(ImageWriteParam.MODE_DEFAULT);
        } catch (UnsupportedOperationException e) {
            LOG.debug("", e);
        }
        try {
            parameters.setCompressionQuality(0.8f);
        } catch (UnsupportedOperationException e) {
            LOG.debug("", e);
        }
    }

    /**
     * Reads a buffered image from the supplied stream and closes the stream afterwards.
     *
     * @param inputStream The stream to read the image from
     * @param formatName The format name
     * @param signaler The optional signaler or <code>null</code>
     * @return The buffered image
     * @throws IOException
     */
    private BufferedImage read(InputStream inputStream, String formatName, ImageTransformationSignaler signaler) throws IOException {
        try {
            return imageIoRead(inputStream, signaler);
        } catch (RuntimeException e) {
            LOG.debug("error reading image from stream for {}", formatName, e);
            return null;
        } finally {
            Streams.close(inputStream);
        }
    }

    /**
     * Reads a buffered image from the supplied stream and closes the stream afterwards, trying to extract meta-data information.
     *
     * @param inputStream The stream to read the image from
     * @param formatName The format name
     * @param maxSize The max. size for an image or less than/equal to 0 (zero) for no size limitation
     * @param maxResolution The max. resolution for an image or less than/equal to 0 (zero) for no resolution limitation
     * @param signaler The optional signaler or <code>null</code>
     * @return The buffered image
     * @throws IOException
     */
    private BufferedImage readAndExtractMetadataFromStream(InputStream inputStream, String formatName, long maxSize, long maxResolution, ImageTransformationSignaler signaler) throws IOException {
        ThresholdFileHolder sink = new ThresholdFileHolder();
        try {
            sink.write(maxSize > 0 ? new CountingInputStream(inputStream, maxSize, IMAGE_SIZE_EXCEEDED_EXCEPTION_CREATOR) : inputStream);
            return readAndExtractMetadataFromFile(sink, formatName, maxSize, maxResolution, signaler);
        } catch (OXException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            }
            throw new IOException("Error accessing file holder", null == cause ? e : cause);
        } finally {
            Streams.close(sink);
        }
    }

    /**
     * Reads a buffered image from the supplied stream and closes the stream afterwards, trying to extract meta-data information.
     *
     * @param imageFile The image file to read from
     * @param formatName The format name
     * @param maxSize The max. size for an image or less than/equal to 0 (zero) for no size limitation
     * @param maxResolution The max. resolution for an image or less than/equal to 0 (zero) for no resolution limitation
     * @param signaler The optional signaler or <code>null</code>
     * @return The buffered image
     */
    private BufferedImage readAndExtractMetadataFromFile(IFileHolder imageFile, String formatName, long maxSize, long maxResolution, ImageTransformationSignaler signaler) throws IOException {
        ImageInputStream imageInputStream = null;
        ImageReader reader = null;
        InputStream inputStream = null;
        try {
            inputStream = imageFile.getStream();
            inputStream = maxSize > 0 ? new CountingInputStream(inputStream, maxSize, IMAGE_SIZE_EXCEEDED_EXCEPTION_CREATOR) : inputStream;
            /*
             * create reader from image input stream
             */
            imageInputStream = getImageInputStream(inputStream);
            reader = getImageReader(imageInputStream, imageFile.getContentType(), imageFile.getName());
            reader.setInput(imageInputStream);
            /*
             * read original image dimensions & check against required dimensions for transformations
             */
            int width = reader.getWidth(0);
            int height = reader.getHeight(0);
            Dimension requiredResolution = getRequiredResolution(transformations, width, height);
            int imageIndex = selectImage(reader, requiredResolution, maxResolution);
            int orientation = readExifOrientation(reader, imageIndex);
            /*
             * prefer a suitable thumbnail in stream if possible when downscaling images
             */
            float preferThumbnailThreshold = preferThumbnailThreshold();
            try {
                if (0 <= preferThumbnailThreshold && reader.hasThumbnails(imageIndex)) {
                    if (null != requiredResolution && (requiredResolution.width < width || requiredResolution.height < height)) {
                        int requiredWidth = (int) (preferThumbnailThreshold * requiredResolution.width);
                        int requiredHeight = (int) (preferThumbnailThreshold * requiredResolution.height);
                        for (int i = 0; i < reader.getNumThumbnails(imageIndex); i++) {
                            int thumbnailWidth = reader.getThumbnailWidth(imageIndex, i);
                            int thumbnailHeight = reader.getThumbnailHeight(imageIndex, i);
                            if (thumbnailWidth >= requiredWidth && thumbnailHeight >= requiredHeight) {
                                LOG.trace("Using thumbnail of {}x{}px (requested: {}x{}px)", I(thumbnailWidth), I(thumbnailHeight), I(requiredResolution.width), I(requiredResolution.height));
                                /*
                                 * use thumbnail & skip any additional scale transformations / compressions
                                 */
                                compress = false;
                                for (Iterator<ImageTransformation> iterator = transformations.iterator(); iterator.hasNext();) {
                                    ImageTransformation transformation = iterator.next();
                                    if (ScaleTransformation.class.isInstance(transformation)) {
                                        iterator.remove();
                                    }
                                }
                                imageInformation = new ImageInformation(orientation, thumbnailWidth, thumbnailHeight);
                                onImageRead(signaler);
                                return reader.readThumbnail(imageIndex, i);
                            }
                        }
                    }
                }
            } catch (IOException e) {
                LOG.debug(e.getMessage(), e);
                // fallback to image transformation
            }
            /*
             * check image size against limitations prior reading source image
             */
            if (0 < maxSize && maxSize < imageFile.getLength()) {
                throw IMAGE_SIZE_EXCEEDED_EXCEPTION_CREATOR.createIOException(imageFile.getLength(), maxSize);
            }
            if (0 < maxResolution) {
                int resolution = height * width;
                if (resolution > maxResolution) {
                    throw createResolutionExceededIOException(maxResolution, resolution);
                }
            }
            imageInformation = new ImageInformation(orientation, width, height);
            onImageRead(signaler);
            return reader.read(imageIndex);
        } catch (RuntimeException e) {
            LOG.debug("error reading image from stream for {}", formatName, e);
            return null;
        } catch (OXException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            }
            throw null == cause ? new IOException(e.getMessage(), e) : new IOException(cause.getMessage(), cause);
        } finally {
            if (null != reader) {
                reader.dispose();
            }
            Streams.close(imageInputStream, inputStream);
        }
    }

    /**
     * Returns a {@link BufferedImage} as the result of decoding a supplied {@code InputStream}.
     *
     * @param in The input stream to read from
     * @param signaler The optional signaler or <code>null</code>
     * @return The resulting {@code BufferedImage} instance
     * @throws IOException If an I/O error occurs
     */
    private BufferedImage imageIoRead(InputStream in, ImageTransformationSignaler signaler) throws IOException {
        onImageRead(signaler);
        return ImageIO.read(in);
    }

    private static void onImageRead(ImageTransformationSignaler signaler) {
        if (null != signaler) {
            try {
                signaler.onImageRead();
            } catch (Exception e) {
                LOG.debug("Signaler could not be called", e);
            }
        }
    }

}
