/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.imageconverter.api;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;

/**
 * {@link IFileItemReadAccess}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public interface IFileItemReadAccess extends Closeable {

    /**
     * @return The {@link InputStream} to read from
     */
    public InputStream getInputStream();

    /**
     * Returns the input {@link File} to read from.</br>
     * A temporary file will be created, if
     * the chosen access option to acquire the {@link IFileItemReadAccess}
     * interface initially doesn't use a file to work on.</br>
     * For performance reasons, care should be taken to use the right access option
     * if planning to use the {@link getInputFile} method.</br>
     * If this method is called, a possibly previously used <code>InputStream</code>
     * object is closed and not usable anymore.</br>
     * In essence, it is best practice to initially decide for
     * {@link AccessOption.FILE_BASED} read access and use either
     * the file object or the stream obect to read content from throughout
     * the lifetime of this access object.
     *
     * @return The {@link File} containing
     *  the actual file content
     * @throws IOException
     */
    public File getInputFile() throws IOException;

    /**
     * @return The creation {@link Date} of the file as Gregorian calendar date
     */
    public Date getCreateDate();

    /**
     * @return The last access {@link Date} of the file in as Gregorian calendar date
     */
    public Date getModificationDate();

    /**
     * @return The length of the existing file item.
     */
    public long getLength();

    /**
     * Returns the value of the FileItem's property with the
     * given key. The key aliases need to be registered once
     * via the {
     *
     * @param key The key to retrieve the value for
     * @return
     */
    public String getKeyValue(final String key) throws FileItemException;
}
