/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.contact.storage.rdb.groupware;

import static com.openexchange.database.Databases.closeSQLStuff;
import static com.openexchange.database.Databases.tableExists;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Map;
import org.slf4j.LoggerFactory;
import com.openexchange.database.Databases;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.PerformParameters;
import com.openexchange.groupware.update.UpdateExceptionCodes;
import com.openexchange.groupware.update.UpdateTaskAdapter;

/**
 * {@link ContactsAccountCreateTableTask}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 * @since v7.10.5
 */
public class ContactsAccountCreateTableTask extends UpdateTaskAdapter {

    /**
     * Initializes a new {@link ContactsAccountCreateTableTask}.
     */
    public ContactsAccountCreateTableTask() {
        super();
    }

    @Override
    public String[] getDependencies() {
        return new String[0];
    }

    @Override
    public void perform(PerformParameters params) throws OXException {
        Connection connection = params.getConnection();
        int rollback = 0;
        try {
            connection.setAutoCommit(false);
            rollback = 1;
            for (Map.Entry<String, String> entry : ContactsAccountCreateTableService.getTablesByName().entrySet()) {
                createTable(connection, entry.getKey(), entry.getValue());
            }
            connection.commit();
            rollback = 2;
        } catch (SQLException e) {
            throw UpdateExceptionCodes.SQL_PROBLEM.create(e, e.getMessage());
        } finally {
            if (rollback > 0) {
                if (rollback == 1) {
                    Databases.rollback(connection);
                }
                Databases.autocommit(connection);
            }
        }
    }

    /**
     * Creates the table with the specified name
     *
     * @param connection The write-able connection
     * @param tableName The table name
     * @param createStatement The create statement
     * @throws SQLException if an SQL error is occurred
     */
    private void createTable(Connection connection, String tableName, String createStatement) throws SQLException {
        if (tableExists(connection, tableName)) {
            LoggerFactory.getLogger(ContactsAccountCreateTableService.class).debug("Table {} already exists, skipping.", tableName);
            return;
        }
        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(createStatement);
            stmt.executeUpdate();
        } finally {
            closeSQLStuff(stmt);
        }
    }
}
