#/bin/bash

function help {
    cat << EOF
${0} --projects=<PROJECT 1>,<PROJECT 2> --backend-dep=<BACKEND RELEASE> --frontend-dep=<FRONTEND RELEASE>

    Optional:
     --check-build-state (prints an overview of the current build state)

    Example:
          Updates the project dependencies
          ${0} --projects=oxio-7.4.0-rev1,oxcs-7.4.0-rev1 --backend-dep=backend-7.4.0-rev10 --frontend-dep=frontend-7.4.0-rev10

          Shows the build state
          ${0} --projects=oxio-7.4.0-rev1,oxcs-7.4.0-rev1 --check-build-state
EOF
}

function error_output {
    set +e
    cat ${1} | \
    grep -v "^$" | \
    grep -v 'WARNING: SSL certificate checks disabled. Connection is insecure!'
    rm -rf ${1}
    set -e
}


NEEDED_TOOLs="\
    osc
    sed
    grep
    mktemp
"
TMP_DIR=$(mktemp -d)

# Check for needed tools
set +e
for tool in ${NEEDED_TOOLs}; do
    which ${tool} &> /dev/null
    if [ ! '0' -eq "${?}" ]; then
        echo "${0}: Error, needed tool not found: ${tool}" >&2
        exit 1
    fi
done
set -e
for argument in $@; do
    case ${argument} in
        --projects=*)
            PROJECTS=${argument##--projects=}
            PROJECTS=$(echo ${PROJECTS} | sed 's|[,;:]| |g')
        ;;
        --backend-dep=*)
            BACKEND_NEW=${argument##--backend-dep=}
        ;;
        --frontend-dep=*)
            FRONTEND_NEW=${argument##--frontend-dep=}
        ;;
        --check-build-state)
            CHECK_BUILD_STATE=yes
        ;;
        *)
            help
            exit 0
        ;;
    esac
done

if [ -z "${PROJECTS}" -o -z "${BACKEND_NEW}" -a  'yes' != "${CHECK_BUILD_STATE}" ]; then
    echo "Error, missing madatory arguments"
    help
    exit 1
fi

for dependency in ${BACKEND_NEW} ${FRONTEND_NEW}; do
    EXSITS=$(osc ls ${dependency} 2>${TMP_DIR}/depencency.error || true)
    error_output ${TMP_DIR}/depencency.error
    if [ -z "${EXSITS}" ]; then
        echo "Error, given dependency ${dependency} do not exists in OBS"
        exit 1
    fi
done


for project in ${PROJECTS}; do
    osc meta prj ${project} > ${TMP_DIR}/${project}.cfg 2>${TMP_DIR}/${project}.error
    error_output ${TMP_DIR}/${project}.error
    BACKEND_OLD=$(cat ${TMP_DIR}/${project}.cfg    | \
                    grep '<path project="backend-' | \
                    sed 's|^.*<path project="||g'  | \
                    sed 's|".*$||g'                | \
                    head -n 1)

    FRONTEND_OLD=$(cat ${TMP_DIR}/${project}.cfg    | \
                    grep '<path project="frontend-' | \
                    sed 's|^.*<path project="||g'   | \
                    sed 's|".*$||g'                 | \
                    head -n 1)

    if [ 'yes' != "${CHECK_BUILD_STATE}" ]; then
        echo "Changing in project: ${project}"

        if [ -z "${BACKEND_OLD}" ]; then
            echo "Error, project configuration of ${project} seams to miss a OX backend dependency"
            exit 1
        else
            echo "Backend dependency from OLD: ${BACKEND_OLD} to NEW: ${BACKEND_NEW}"
            sed -i "s|${BACKEND_OLD}|${BACKEND_NEW}|g" ${TMP_DIR}/${project}.cfg
        fi

        if [ -z "${FRONTEND_OLD}" -o -z "${FRONTEND_NEW}" ]; then
            echo "Warning, project configuration of ${project} seams to miss a OX Frontend dependency"
        else
            echo "Frontend dependency from OLD: ${FRONTEND_OLD} to NEW: ${FRONTEND_NEW}"
            sed -i "s|${FRONTEND_OLD}|${FRONTEND_NEW}|g" ${TMP_DIR}/${project}.cfg
        fi
        osc meta prj ${project} --file=${TMP_DIR}/${project}.cfg 2>${TMP_DIR}/${project}.error
        error_output ${TMP_DIR}/${project}.error
    else
        ARCHITECTURES=$(cat ${TMP_DIR}/${project}.cfg | \
                       grep '<arch>'                  | \
                       sed 's|^.*<arch>||g'           | \
                       sed 's|<.*$||g' | sort -u)

        REPOSITORIES=$(cat ${TMP_DIR}/${project}.cfg | \
                      grep 'repository='             | \
                      sed 's|^.*repository="||g'     | \
                      sed 's|".*$||g'| sort -u)

        PACKAGES=$(osc ls ${project} 2>${TMP_DIR}/${project}.error)
        error_output ${TMP_DIR}/${project}.error

        for pkg in ${PACKAGES}; do
            for repo in ${REPOSITORIES}; do
                for arch in ${ARCHITECTURES}; do
                    DETAILS=$(osc jobhistory ${project} ${pkg} ${repo} ${arch} 2>${TMP_DIR}/${project}.error | \
                        tail -n 1 | \
                        awk '{printf $6"	"$1"	"$2"	"$4"	"$5}')
                    echo "${DETAILS}	${arch}	${project}		${repo}	${pkg}" >>${TMP_DIR}/${project}.status
                    error_output ${TMP_DIR}/${project}.error
                done
            done
        done
        echo  "" >> ${TMP_DIR}/${project}.status
    fi
done
if [ 'yes' == "${CHECK_BUILD_STATE}" ]; then
    echo "Buildstate	Date		Time		Type	What	Arch	Project				Repo		Package"
    echo "------------------------------------------------------------------------------------------------------------------------------------------"
    cat ${TMP_DIR}/*.status
fi
rm -rf ${TMP_DIR}
