/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.hazelcast.ssl.utils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * {@link StringUtils} - String utility for Hazelcast SSL.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.10.6
 */
public class StringUtils {

    /**
     * Initializes a new {@link StringUtils}.
     */
    private StringUtils() {
        super();
    }

    /**
     * Checks for an empty string.
     *
     * @param str The string
     * @return <code>true</code> if input is <code>null</code>, empty or only consists of white-space characters; else <code>false</code>
     */
    public static boolean isEmpty(final String str) {
        if (null == str) {
            return true;
        }
        final int len = str.length();
        boolean isWhitespace = true;
        for (int i = len; isWhitespace && i-- > 0;) {
            isWhitespace = isWhitespace(str.charAt(i));
        }
        return isWhitespace;
    }

    /**
     * High speed test for whitespace! Faster than the java one (from some testing).
     *
     * @param c The character to check
     * @return <code>true</code> if the indicated character is whitespace; otherwise <code>false</code>
     */
    private static boolean isWhitespace(final char c) {
        switch (c) {
            case 9: // 'unicode: 0009
            case 10: // 'unicode: 000A'
            case 11: // 'unicode: 000B'
            case 12: // 'unicode: 000C'
            case 13: // 'unicode: 000D'
            case 28: // 'unicode: 001C'
            case 29: // 'unicode: 001D'
            case 30: // 'unicode: 001E'
            case 31: // 'unicode: 001F'
            case ' ': // Space
                // case Character.SPACE_SEPARATOR:
                // case Character.LINE_SEPARATOR:
            case Character.PARAGRAPH_SEPARATOR:
                return true;
            default:
                return false;
        }
    }

    private static final char[] HEX_CHARS = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

    /**
     * Turns array of bytes into string representing each byte as unsigned hex number.
     *
     * @param hash Array of bytes to convert to hex-string
     * @return Generated hex string
     */
    private static char[] asHexChars(byte[] hash) {
        if (hash == null) {
            return null;
        }
        int length = hash.length;
        if (length <= 0) {
            return new char[0];
        }
        char[] buf = new char[length << 1];
        for (int i = 0, x = 0; i < length; i++) {
            buf[x++] = HEX_CHARS[(hash[i] >>> 4) & 0xf];
            buf[x++] = HEX_CHARS[hash[i] & 0xf];
        }
        return buf;
    }

    /**
     * Turns array of bytes into string representing each byte as unsigned hex number.
     *
     * @param hash Array of bytes to convert to hex-string
     * @return Generated hex string
     */
    public static String asHex(byte[] hash) {
        char[] chars = asHexChars(hash);
        return chars == null ? null : new String(chars);
    }

    /**
     * Splits given string by comma separator.
     *
     * @param s The string to split
     * @return The split string
     */
    public static String[] splitByComma(final String s) {
        return splitBy(s, ',', true);
    }

    /**
     * Splits given string by specified character.
     *
     * @param s The string to split
     * @param delim The delimiter to split by
     * @param trimMatches <code>true</code> to trim tokens; otherwise <code>false</code>
     * @return The split string
     */
    public static String[] splitBy(String s, char delim, boolean trimMatches) {
        if (null == s) {
            return null;
        }
        int length = s.length();
        if (length == 0) {
            return new String[] { trimMatches ? s.trim() : s };
        }

        int pos = s.indexOf(delim, 0);
        if (pos < 0) {
            return new String[] { trimMatches ? s.trim() : s };
        }

        List<String> matchList = new ArrayList<String>();
        collectMatches(pos, s, length, delim, trimMatches, matchList);
        return matchList.toArray(new String[matchList.size()]);
    }

    private static <C extends Collection<String>> void collectMatches(int firstPos, String s, int length, char delim, boolean trimMatches, C matchList) {
        int pos = firstPos;
        int prevPos = 0;
        do {
            matchList.add(trimMatches ? s.substring(prevPos, pos).trim() : s.substring(prevPos, pos));
            prevPos = pos + 1;
            pos = prevPos < length ? s.indexOf(delim, prevPos) : -1;
        } while (pos >= 0);
        if (prevPos <= length) {
            matchList.add(trimMatches ? s.substring(prevPos).trim() : s.substring(prevPos));
        }
    }

}
