/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.logback.extensions.logstash;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang.time.FastDateFormat;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonGenerator.Feature;
import ch.qos.logback.classic.spi.CallerData;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.classic.spi.ThrowableProxyUtil;
import ch.qos.logback.core.encoder.EncoderBase;
import ch.qos.logback.core.status.ErrorStatus;

/**
 * {@link LogstashEncoder}. Encodes {@link ILoggingEvent} objects as JSON objects.
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class LogstashEncoder extends EncoderBase<ILoggingEvent> {

    public static final FastDateFormat LOGSTASH_TIMEFORMAT = FastDateFormat.getInstance("yyyy-MM-dd'T'HH:mm:ss.SSSZZ");

    private final List<CustomField> customFields;

    /**
     * Initialises a new {@link LogstashEncoder}.
     */
    public LogstashEncoder() {
        super();
        customFields = new ArrayList<CustomField>();
    }

    /**
     * Adds the specified {@link CustomField} to the {@link LogstashFormatter}
     *
     * @param customField the {@link CustomField} to add
     */
    public void addCustomField(CustomField customField) {
        customFields.add(customField);
    }

    /*
     * (non-Javadoc)
     * 
     * @see ch.qos.logback.core.encoder.Encoder#headerBytes()
     */
    @Override
    public byte[] headerBytes() {
        // no-op
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see ch.qos.logback.core.encoder.Encoder#encode(java.lang.Object)
     */
    @Override
    public byte[] encode(ILoggingEvent event) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        JsonGenerator generator = null;
        try {
            generator = new JsonFactory().createGenerator(baos);
            generator.configure(Feature.FLUSH_PASSED_TO_STREAM, false);
            generator.writeStartObject();

            generator.writeStringField(LogstashFieldName.timestamp.getLogstashName(), LOGSTASH_TIMEFORMAT.format(event.getTimeStamp()));
            generator.writeNumberField(LogstashFieldName.version.getLogstashName(), 1);

            // Logger
            generator.writeStringField(LogstashFieldName.level.getLogstashName(), event.getLevel().levelStr);
            generator.writeStringField(LogstashFieldName.logger.getLogstashName(), event.getLoggerName());

            // App specific
            generator.writeStringField(LogstashFieldName.thread.getLogstashName(), event.getThreadName());
            generator.writeStringField(LogstashFieldName.message.getLogstashName(), event.getFormattedMessage());

            generator.writeNumberField(LogstashFieldName.line.getLogstashName(), getLineNumber(event));
            generator.writeStringField(LogstashFieldName.clazz.getLogstashName(), getFullyQualifiedName(event));

            if (event.getMarker() != null) {
                generator.writeStringField(LogstashFieldName.marker.getLogstashName(), event.getMarker().getName());
            }

            // Stacktraces
            if (event.getThrowableProxy() != null) {
                generator.writeStringField(LogstashFieldName.stacktrace.getLogstashName(), ThrowableProxyUtil.asString(event.getThrowableProxy()));
            }

            // MDC
            Map<String, String> mdc = event.getMDCPropertyMap();
            for (String key : mdc.keySet()) {
                generator.writeFieldName(key);
                generator.writeObject(mdc.get(key));
            }

            for (CustomField customField : customFields) {
                generator.writeFieldName(customField.getKey());
                generator.writeObject(customField.getValue());
            }

            generator.writeEndObject();
            generator.flush();
        } catch (IOException e) {
            addStatus(new ErrorStatus("An I/O error occurred during event encoding", this, e));
        } finally {
            if (generator != null) {
                try {
                    generator.close();
                } catch (IOException e) {
                    addStatus(new ErrorStatus("An I/O error occurred while closing the JsonGenerator", this, e));
                }
            }
        }
        return baos.toByteArray();
    }

    /*
     * (non-Javadoc)
     * 
     * @see ch.qos.logback.core.encoder.Encoder#footerBytes()
     */
    @Override
    public byte[] footerBytes() {
        // no-op
        return null;
    }

    /**
     * Returns the line number of the class that the log event was initiated from
     * 
     * @param event The {@link ILoggingEvent}
     * @return the line number of the class that the log event was initiated from or '-1' if none available
     */
    private int getLineNumber(ILoggingEvent event) {
        StackTraceElement[] cda = event.getCallerData();
        if (cda != null && cda.length > 0) {
            return cda[0].getLineNumber();
        }
        return CallerData.LINE_NA;
    }

    /**
     * Returns the fully qualified name of the logger
     * 
     * @param event The {@link ILoggingEvent}
     * @return the fully qualified name of the logger or '?' if none available
     */
    private String getFullyQualifiedName(ILoggingEvent event) {
        StackTraceElement[] cda = event.getCallerData();
        if (cda != null && cda.length > 0) {
            return cda[0].getClassName();
        }
        return CallerData.NA;
    }
}
