package org.apache.jcs.engine;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.apache.jcs.engine.behavior.IElementAttributes;
import org.apache.jcs.engine.control.event.behavior.IElementEventHandler;

/**
 * This it the element attribute descriptor class. Each element in the cache has
 * an ElementAttribute object associated with it. An ElementAttributes object
 * can be associated with an element in 3 ways:
 * <ol>
 * <li>When the item is put into the cache, you can associate an element
 * attributes object.</li>
 * <li>If not attributes object is include when the element is put into the
 * cache, then the default attributes for the region will be used.</li>
 * <li>The element attributes can be reset. This effectively results in a
 * retrieval followed by a put. Hence, this is the same as 1.</li>
 * </ol>
 * @version $Id: ILateralCacheTCPListener.java,v 1.2 2002/01/18 22:08:26
 */
public class ElementAttributes
    implements IElementAttributes, Serializable, Cloneable
{
    private static final long serialVersionUID = 7814990748035017441L;

    /**
     * Can this item be flushed to disk
     */
    public final java.util.concurrent.atomic.AtomicBoolean IS_SPOOL = new java.util.concurrent.atomic.AtomicBoolean(true);

    /**
     * Is this item laterally distributable
     */
    public final java.util.concurrent.atomic.AtomicBoolean IS_LATERAL = new java.util.concurrent.atomic.AtomicBoolean(true);

    /**
     * Can this item be sent to the remote cache
     */
    public final java.util.concurrent.atomic.AtomicBoolean IS_REMOTE = new java.util.concurrent.atomic.AtomicBoolean(true);

    /**
     * You can turn off expiration by setting this to true. This causes the
     * cache to bypass both max life and idle time expiration.
     */
    public final java.util.concurrent.atomic.AtomicBoolean IS_ETERNAL = new java.util.concurrent.atomic.AtomicBoolean(true);

    /**
     * The object version. This is currently not used.
     */
    public final java.util.concurrent.atomic.AtomicLong version = new java.util.concurrent.atomic.AtomicLong(0);

    /**
     * Max life seconds
     */
    public final java.util.concurrent.atomic.AtomicLong maxLifeSeconds = new java.util.concurrent.atomic.AtomicLong(-1);

    /**
     * The maximum time an entry can be idle. Setting this to -1 causes the idle
     * time check to be ignored.
     */
    public final java.util.concurrent.atomic.AtomicLong maxIdleTimeSeconds = new java.util.concurrent.atomic.AtomicLong(-1);

    /**
     * The byte size of the field. Must be manually set.
     */
    public final java.util.concurrent.atomic.AtomicInteger size = new java.util.concurrent.atomic.AtomicInteger(0);

    /**
     * The creation time. This is used to enforce the max life.
     */
    public final java.util.concurrent.atomic.AtomicLong createTime = new java.util.concurrent.atomic.AtomicLong(0);

    /**
     * The last access time. This is used to enforce the max idel time.
     */
    public final java.util.concurrent.atomic.AtomicLong lastAccessTime = new java.util.concurrent.atomic.AtomicLong(0);

    /**
     * The list of Event handlers to use. This is transient, since the event
     * handlers cannot usually be serialized. This means that you cannot attach
     * a post serialization event to an item.
     * <p>
     * TODO we need to check that when an item is passed to a non-local cache
     * that if the local cache had a copy with event handlers, that those
     * handlers are used.
     */
    public transient Queue<IElementEventHandler> eventHandlers;

    /**
     * Constructor for the IElementAttributes object
     */
    public ElementAttributes()
    {
        long now = System.currentTimeMillis();
        this.createTime.set(now);
        this.lastAccessTime.set(now);
    }

    /**
     * Constructor for the IElementAttributes object
     * <p>
     * @param attr
     */
    protected ElementAttributes( ElementAttributes attr )
    {
        IS_ETERNAL.set(attr.IS_ETERNAL.get());

        // waterfal onto disk, for pure disk set memory to 0
        IS_SPOOL.set(attr.IS_SPOOL.get());

        // lateral
        IS_LATERAL.set(attr.IS_LATERAL.get());

        // central rmi store
        IS_REMOTE.set(attr.IS_REMOTE.get());

        maxLifeSeconds.set(attr.maxLifeSeconds.get());
        // timetolive
        maxIdleTimeSeconds.set(attr.maxIdleTimeSeconds.get());
        size.set(attr.size.get());
    }

    /**
     * Copies the attributes, including references to event handlers.
     * <p>
     * @return a copy of the Attributes
     */
    @Override
    public IElementAttributes copy()
    {
        try
        {
            // need to make this more efficient. Just want to insure
            // a proper copy
            ElementAttributes attr = new ElementAttributes();
            attr.setIdleTime( this.getIdleTime() );
            attr.setIsEternal( this.getIsEternal() );
            attr.setIsLateral( this.getIsLateral() );
            attr.setIsRemote( this.getIsRemote() );
            attr.setIsSpool( this.getIsSpool() );
            attr.setMaxLifeSeconds( this.getMaxLifeSeconds() );
            attr.addElementEventHandlers( this.eventHandlers );
            return attr;
        }
        catch ( Exception e )
        {
            return new ElementAttributes();
        }
    }

    /**
     * Deep clone the attributes.
     * <p>
     * @return a clone of these attributes
     */
    public Object clone2()
    {
        try
        {
            ByteArrayOutputStream baos = new ByteArrayOutputStream( 100 );
            ObjectOutputStream oos = new ObjectOutputStream( baos );
            oos.writeObject( this );
            byte buf[] = baos.toByteArray();
            oos.close();

            // deserialize byte array into ArrayList

            ByteArrayInputStream bais = new ByteArrayInputStream( buf );
            ObjectInputStream ois = new ObjectInputStream( bais );
            ElementAttributes attr = (ElementAttributes) ois.readObject();
            ois.close();

            attr.createTime.set(System.currentTimeMillis());
            return attr;
        }
        catch ( Exception e )
        {
            // swallow
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#setVersion(long)
     */
    @Override
    public void setVersion( long version )
    {
        this.version.set(version);
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#setMaxLifeSeconds(long)
     */
    @Override
    public void setMaxLifeSeconds( long mls )
    {
        this.maxLifeSeconds.set(mls);
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#getMaxLifeSeconds()
     */
    @Override
    public long getMaxLifeSeconds()
    {
        return this.maxLifeSeconds.get();
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#setIdleTime(long)
     */
    @Override
    public void setIdleTime( long idle )
    {
        this.maxIdleTimeSeconds.set(idle);
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#setSize(int)
     */
    @Override
    public void setSize( int size )
    {
        this.size.set(size);
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#getSize()
     */
    @Override
    public int getSize()
    {
        return size.get();
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#getCreateTime()
     */
    @Override
    public long getCreateTime()
    {
        return createTime.get();
    }

    /**
     * Sets the createTime attribute of the IElementAttributes object
     */
    public void setCreateTime()
    {
        createTime.set(System.currentTimeMillis());
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#getVersion()
     */
    @Override
    public long getVersion()
    {
        return version.get();
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#getIdleTime()
     */
    @Override
    public long getIdleTime()
    {
        return this.maxIdleTimeSeconds.get();
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#getTimeToLiveSeconds()
     */
    @Override
    public long getTimeToLiveSeconds()
    {
        long now = System.currentTimeMillis();
        return ( ( this.getCreateTime() + ( this.getMaxLifeSeconds() * 1000 ) ) - now ) / 1000;
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#getLastAccessTime()
     */
    @Override
    public long getLastAccessTime()
    {
        return this.lastAccessTime.get();
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#setLastAccessTimeNow()
     */
    @Override
    public void setLastAccessTimeNow()
    {
        this.lastAccessTime.set(System.currentTimeMillis());
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#getIsSpool()
     */
    @Override
    public boolean getIsSpool()
    {
        return this.IS_SPOOL.get();
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#setIsSpool(boolean)
     */
    @Override
    public void setIsSpool( boolean val )
    {
        this.IS_SPOOL.set(val);
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#getIsLateral()
     */
    @Override
    public boolean getIsLateral()
    {
        return this.IS_LATERAL.get();
    }

    /*
     * (non-Javadoc)
     * @see org.apache.jcs.engine.behavior.IElementAttributes#setIsLateral(boolean)
     */
    @Override
    public void setIsLateral( boolean val )
    {
        this.IS_LATERAL.set(val);
    }

    /**
     * Can this item be sent to the remote cache
     * @return true if the item can be sent to a remote auxiliary
     */
    @Override
    public boolean getIsRemote()
    {
        return this.IS_REMOTE.get();
    }

    /**
     * Sets the isRemote attribute of the ElementAttributes object
     * @param val
     *            The new isRemote value
     */
    @Override
    public void setIsRemote( boolean val )
    {
        this.IS_REMOTE.set(val);
    }

    /**
     * You can turn off expiration by setting this to true. The max life value
     * will be ignored.
     * <p>
     * @return true if the item cannot expire.
     */
    @Override
    public boolean getIsEternal()
    {
        return this.IS_ETERNAL.get();
    }

    /**
     * Sets the isEternal attribute of the ElementAttributes object. True means
     * that the item should never expire. If can still be removed if it is the
     * least recently used, and you are using the LRUMemory cache. it just will
     * not be filtered for expiration by the cache hub.
     * <p>
     * @param val
     *            The new isEternal value
     */
    @Override
    public void setIsEternal( boolean val )
    {
        this.IS_ETERNAL.set(val);
    }

    /**
     * Adds a ElementEventHandler. Handler's can be registered for multiple
     * events. A registered handler will be called at every recognized event.
     * <p>
     * The alternative would be to register handlers for each event. Or maybe
     * The handler interface should have a method to return whether it cares
     * about certain events.
     * <p>
     * @param eventHandler
     *            The ElementEventHandler to be added to the list.
     */
    @Override
    public void addElementEventHandler( IElementEventHandler eventHandler )
    {
        // lazy here, no concurrency problems expected
        Queue<IElementEventHandler> eventHandlers = this.eventHandlers;
        if ( eventHandlers == null )
        {
            eventHandlers = new ConcurrentLinkedQueue<>();
            this.eventHandlers = eventHandlers;
        }
        eventHandlers.add( eventHandler );
    }

    /**
     * Sets the eventHandlers of the IElementAttributes object.
     * <p>
     * This add the references to the local list. Subsequent changes in the
     * caller's list will not be reflected.
     * <p>
     * @param eventHandlers
     *            List of IElementEventHandler objects
     */
    @Override
    public void addElementEventHandlers( Queue<IElementEventHandler> eventHandlers )
    {
        if ( eventHandlers == null )
        {
            return;
        }

        for ( Iterator<IElementEventHandler> iter = eventHandlers.iterator(); iter.hasNext(); )
        {
            addElementEventHandler( iter.next() );
        }
    }

    /**
     * Gets the elementEventHandlers. Returns null if none exist. Makes checking
     * easy.
     * <p>
     * @return The elementEventHandlers List of IElementEventHandler objects
     */
    @Override
    public Queue<IElementEventHandler> getElementEventHandlers()
    {
        return this.eventHandlers;
    }

    /**
     * For logging and debugging the element IElementAttributes.
     * <p>
     * @return String info about the values.
     */
    @Override
    public String toString()
    {
        StringBuffer dump = new StringBuffer();

        dump.append( "[ IS_LATERAL = " ).append( IS_LATERAL );
        dump.append( ", IS_SPOOL = " ).append( IS_SPOOL );
        dump.append( ", IS_REMOTE = " ).append( IS_REMOTE );
        dump.append( ", IS_ETERNAL = " ).append( IS_ETERNAL );
        dump.append( ", MaxLifeSeconds = " ).append( this.getMaxLifeSeconds() );
        dump.append( ", IdleTime = " ).append( this.getIdleTime() );
        dump.append( ", CreateTime = " ).append( this.getCreateTime() );
        dump.append( ", LastAccessTime = " ).append( this.getLastAccessTime() );
        dump.append( ", getTimeToLiveSeconds() = " ).append( String.valueOf( getTimeToLiveSeconds() ) );
        dump.append( ", createTime = " ).append( String.valueOf( createTime ) ).append( " ]" );

        return dump.toString();
    }
}
