/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.share.impl.cleanup;

import static com.openexchange.java.Autoboxing.I;
import static com.openexchange.java.Autoboxing.L;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.context.ContextService;
import com.openexchange.exception.Category;
import com.openexchange.exception.OXException;
import com.openexchange.server.ServiceLookup;
import com.openexchange.share.ShareExceptionCodes;

/**
 * {@link PeriodicCleaner}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 * @since v7.8.0
 */
public class PeriodicCleaner implements Runnable {

    private static final Logger LOG = LoggerFactory.getLogger(PeriodicCleaner.class);

    private final ServiceLookup services;
    private final long guestExpiry;
    private final AtomicBoolean active;

    /**
     * Initializes a new {@link PeriodicCleaner}.
     *
     * @param services A service lookup reference
     * @param guestExpiry the timespan (in milliseconds) after which an unused guest user can be deleted permanently
     */
    public PeriodicCleaner(final ServiceLookup services, long guestExpiry) {
        super();
        this.services = services;
        this.guestExpiry = guestExpiry;
        this.active = new AtomicBoolean(true);
    }

    @Override
    public void run() {
        long start = System.currentTimeMillis();
        try {
            List<Integer> distinctContextsPerSchema = services.getService(ContextService.class).getDistinctContextsPerSchema();
            int size = distinctContextsPerSchema.size();
            LOG.info("Periodic share cleanup task starting, going to check {} schemas...", I(size));
            long logTimeDistance = TimeUnit.SECONDS.toMillis(10);
            long lastLogTime = start;
            int i = 0;
            for (Integer representativeCtxId : distinctContextsPerSchema) {
                int representativeContextId = representativeCtxId.intValue();
                for (int retry = 0; retry < 3; retry++) {
                    if (false == active.get()) {
                        LOG.info("Periodic share cleanup task stopping.");
                        return;
                    }
                    long now = System.currentTimeMillis();
                    if (now > lastLogTime + logTimeDistance) {
                        LOG.info("Periodic share cleanup task {}% finished ({}/{}).",
                            I(i * 100 / size), I(i), I(size));
                        lastLogTime = now;
                    }
                    try {
                        cleanupSchema(representativeContextId);
                        break;
                    } catch (OXException e) {
                        if (Category.CATEGORY_TRY_AGAIN.equals(e.getCategory()) && retry < 3) {
                            long delay = 10000 + retry * 20000;
                            LOG.debug("Error during periodic share cleanup task for schema of context {}: {}; trying again in {}ms...", I(representativeContextId), e.getMessage(), L(delay));
                            Thread.sleep(delay);
                        } else {
                            LOG.error("Error during periodic share cleanup task for schema of context {}: {}", I(representativeContextId), e.getMessage(), e);
                            break;
                        }
                    }
                }
                i++;
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.warn("Interrupted during periodic share cleanup task: {}", e.getMessage(), e);
        } catch (Exception e) {
            LOG.error("Error during periodic share cleanup task: {}", e.getMessage(), e);
        }
        LOG.info("Periodic share cleanup task finished after {}ms.", L(System.currentTimeMillis() - start));
    }

    /**
     * Stops all background processing by signaling termination flag.
     */
    public void stop() {
        active.set(false);
    }

    /**
     * Synchronously cleans obsolete shares and corresponding guest user remnants for a schema.
     *
     * @param services A service lookup reference
     * @param representativeContextId The context ID located in target schema
     * @param guestExpiry the time span (in milliseconds) after which an unused guest user can be deleted permanently
     */
    private void cleanupSchema(int representativeContextId) throws OXException {
        // Execute schema- and resulting guest cleanup tasks in current thread
        try {
            Map<Integer, List<GuestCleanupTask>> guestCleanupTasks = new SchemaCleanupTask(services, representativeContextId, guestExpiry).call();
            for (Map.Entry<Integer, List<GuestCleanupTask>> guestCleanupTasksEntry : guestCleanupTasks.entrySet()) {
                if (false == active.get()) {
                    return;
                }
                for (GuestCleanupTask guestCleanupTask : guestCleanupTasksEntry.getValue()) {
                    if (false == active.get()) {
                        return;
                    }
                    guestCleanupTask.call();
                }
            }
        } catch (Exception e) {
            if (OXException.class.isInstance(e)) {
                throw (OXException) e;
            }
            throw ShareExceptionCodes.UNEXPECTED_ERROR.create(e, "Unexpected error during cleanup");
        }
    }

}
