/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.groupware.filestore;

import static com.openexchange.java.Autoboxing.I;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import com.openexchange.database.Databases;
import com.openexchange.exception.OXException;
import com.openexchange.server.impl.DBPool;

public class RdbFilestoreStorage extends FilestoreStorage {

    @Override
    public Filestore getFilestore(int id) throws OXException {
        Connection con = DBPool.pickup();
        try {
            return getFilestore(con, id);
        } finally {
            DBPool.closeReaderSilent(con);
        }
    }

    @Override
    public Filestore getFilestore(Connection con, int id) throws OXException {
        if (null == con) {
            return getFilestore(id);
        }

        PreparedStatement stmt = null;
        ResultSet result = null;
        try {
            stmt = con.prepareStatement("SELECT uri, size, max_context FROM filestore WHERE id = ?");
            stmt.setInt(1,id);
            result = stmt.executeQuery();
            if (!result.next()) {
                throw FilestoreExceptionCodes.NO_SUCH_FILESTORE.create(I(id));
            }

            FilestoreImpl filestore = new FilestoreImpl();
            filestore.setId(id);
            String tmp = null;
            try {
                tmp = result.getString("uri");
                filestore.setUri(new URI(tmp));
            } catch (URISyntaxException e) {
                throw FilestoreExceptionCodes.URI_CREATION_FAILED.create(e, tmp);
            }

            filestore.setSize(result.getLong("size"));
            filestore.setMaxContext(result.getLong("max_context"));
            return filestore;
        } catch (SQLException e) {
            throw FilestoreExceptionCodes.SQL_PROBLEM.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(result, stmt);
        }
    }

    @Override
    public Filestore getFilestore(URI uri) throws OXException {
        Connection con = DBPool.pickup();
        try {
            return getFilestore(con, uri);
        } finally {
            DBPool.closeReaderSilent(con);
        }
    }

    /**
     * Gets the file store for given URI using specified connection.
     *
     * @param con The connection to use
     * @param uri The URI to resolve
     * @return The associated file store
     * @throws OXException If file store cannot be resolved
     */
    public Filestore getFilestore(Connection con, URI uri) throws OXException {
        if (null == con) {
            return getFilestore(uri);
        }

        PreparedStatement stmt = null;
        ResultSet result = null;
        try {
            stmt = con.prepareStatement("SELECT id, uri, size, max_context FROM filestore WHERE (? LIKE CONCAT(uri, '%'))");
            stmt.setString(1, uri.toString());
            result = stmt.executeQuery();
            if (!result.next()) {
                throw FilestoreExceptionCodes.NO_SUCH_FILESTORE.create(uri);
            }

            FilestoreImpl filestore = new FilestoreImpl();
            filestore.setId(result.getInt(1));
            String tmp = null;
            try {
                tmp = result.getString("uri");
                filestore.setUri(new URI(tmp));
            } catch (URISyntaxException e) {
                throw FilestoreExceptionCodes.URI_CREATION_FAILED.create(e, tmp);
            }

            filestore.setSize(result.getLong("size"));
            filestore.setMaxContext(result.getLong("max_context"));
            return filestore;
        } catch (SQLException e) {
            throw FilestoreExceptionCodes.SQL_PROBLEM.create(e, e.getMessage());
        } finally {
            Databases.closeSQLStuff(result, stmt);
        }

    }
}
