/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.contact.picture.impl.finder;

import java.util.Iterator;
import java.util.Set;
import com.openexchange.contact.ContactService;
import com.openexchange.contact.SortOptions;
import com.openexchange.contact.SortOrder;
import com.openexchange.contact.picture.PictureSearchData;
import com.openexchange.contact.picture.impl.ContactPictureUtil;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.contact.helpers.ContactField;
import com.openexchange.groupware.container.Contact;
import com.openexchange.groupware.search.ContactSearchObject;
import com.openexchange.groupware.search.Order;
import com.openexchange.java.Streams;
import com.openexchange.java.Strings;
import com.openexchange.session.Session;
import com.openexchange.tools.arrays.Arrays;
import com.openexchange.tools.iterator.SearchIterator;

/**
 * {@link ContactMailFinder} - Finds picture based on the provided mail addresses
 *
 * @author <a href="mailto:daniel.becker@open-xchange.com">Daniel Becker</a>
 * @since v7.10.1
 */
public class ContactMailFinder extends AbstractContactFinder {

    /**
     * Initializes a new {@link ContactMailFinder}.
     *
     * @param contactService The {@link ContactService}
     */
    public ContactMailFinder(ContactService contactService) {
        super(contactService);
    }
    // ---------------------------------------------------------------------------------------------

    @Override
    public Contact getContact(Session session, PictureSearchData data, ContactField... fields) throws OXException {
        if (data.hasEmail()) {
            return findContactByMail(session, data, fields);
        }
        return null;
    }

    @Override
    public PictureSearchData modfiyResult(Contact contact) {
        // Do nothing
        return PictureSearchData.EMPTY_DATA;
    }

    @Override
    void handleException(PictureSearchData data, OXException e) {
        LOGGER.debug("Unable to get contact for mail addresses {}.", data.getEmails(), e);
    }
    // ---------------------------------------------------------------------------------------------

    /**
     * Searches for a contact via its mail address.
     *
     * @param session The {@link Session}
     * @param data The {@link PictureSearchData}
     * @param fields The {@link ContactField}s that should be retrieved
     * @return The {@link Contact} or <code>null</code>
     * @throws OXException If the contact could not be found
     */
    private Contact findContactByMail(Session session, PictureSearchData data, ContactField... fields) throws OXException {
        // Make sure fields for sorting are available
        if (false == Arrays.contains(fields, ContactField.OBJECT_ID)) {
            Arrays.add(fields, ContactField.OBJECT_ID);
        }
        if (false == Arrays.contains(fields, ContactField.FOLDER_ID)) {
            Arrays.add(fields, ContactField.FOLDER_ID);
        }

        Set<String> emails = data.getEmails();
        for (Iterator<String> iterator = emails.iterator(); iterator.hasNext();) {
            String email = iterator.next();
            if (Strings.isEmpty(email) || false == ContactPictureUtil.isValidMailAddress(email)) {
                // skip empty email addresses
                continue;
            }

            ContactSearchObject cso = new ContactSearchObject();
            cso.setAllEmail(email);
            cso.setOrSearch(true);
            cso.setHasImage(true);
            cso.setExactMatch(true);

            SearchIterator<Contact> result = null;
            try {
                // Search from system folder (e.g. GAB, if accessible) to private folders, by oldest contacts first
                result = contactService.searchContacts(session, cso, fields, new SortOptions(new SortOrder(ContactField.FOLDER_ID, Order.ASCENDING), new SortOrder(ContactField.OBJECT_ID, Order.ASCENDING)));
                if (result == null) {
                    continue;
                }

                while (result.hasNext()) {
                    Contact contact = result.next();
                    return contact;
                }
            } finally {
                Streams.close(result);
            }

        }
        return null;
    }

}
