/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.chronos.provider.caching.internal.response;

import static com.openexchange.chronos.common.CalendarUtils.getFlags;
import static com.openexchange.chronos.common.CalendarUtils.isInRange;
import static com.openexchange.chronos.common.CalendarUtils.isSeriesMaster;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;
import com.openexchange.chronos.Event;
import com.openexchange.chronos.EventField;
import com.openexchange.chronos.provider.caching.basic.BasicCachingCalendarAccess;
import com.openexchange.chronos.provider.caching.internal.Services;
import com.openexchange.chronos.service.CalendarParameters;
import com.openexchange.chronos.service.RecurrenceIterator;
import com.openexchange.chronos.service.RecurrenceService;
import com.openexchange.chronos.service.SearchOptions;
import com.openexchange.chronos.storage.CalendarStorage;
import com.openexchange.chronos.storage.operation.OSGiCalendarStorageOperation;
import com.openexchange.exception.OXException;
import com.openexchange.java.Strings;

/**
 * {@link AccountResponseGenerator}
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @since v7.10.0
 */
public class AccountResponseGenerator extends ResponseGenerator {

    public AccountResponseGenerator(BasicCachingCalendarAccess cachedCalendarAccess) {
        super(cachedCalendarAccess);
    }

    public List<Event> generate() throws OXException {
        CalendarParameters parameters = cachedCalendarAccess.getParameters();
        Date from = getFrom(parameters);
        Date until = getUntil(parameters);
        TimeZone timeZone = getTimeZone(parameters, cachedCalendarAccess.getSession());
        SearchOptions searchOptions = new SearchOptions(parameters);
        return new OSGiCalendarStorageOperation<List<Event>>(Services.getServiceLookup(), this.cachedCalendarAccess.getSession().getContextId(), this.cachedCalendarAccess.getAccount().getAccountId()) {

            @Override
            protected List<Event> call(CalendarStorage storage) throws OXException {
                CalendarParameters parameters = cachedCalendarAccess.getParameters();
                EventField[] fields = getFields(parameters.get(CalendarParameters.PARAMETER_FIELDS, EventField[].class), EventField.FOLDER_ID);
                List<Event> events = storage.getEventStorage().searchEvents(null, searchOptions, fields);
                List<Event> enhancedEvents = storage.getUtilities().loadAdditionalEventData(cachedCalendarAccess.getAccount().getUserId(), events, fields);
                List<Event> allEvents = new ArrayList<>();
                for (Event event : enhancedEvents) {
                    event.setFlags(getFlags(event, cachedCalendarAccess.getAccount().getUserId()));
                    if (isSeriesMaster(event) && Strings.isNotEmpty(event.getRecurrenceRule())) {
                        RecurrenceIterator<Event> iterator = Services.getService(RecurrenceService.class).iterateEventOccurrences(event, from, until);
                        if (isResolveOccurrences(parameters)) {
                            while (iterator.hasNext()) {
                                Event occurrence = iterator.next();
                                if (isInRange(occurrence, from, until, timeZone)) {
                                    allEvents.add(occurrence);
                                }
                            }
                        } else if (iterator.hasNext()) {
                            allEvents.add(event);
                        }
                    } else if (isInRange(event, from, until, timeZone)) {
                        allEvents.add(event);
                    }
                }
                return allEvents;
            }
        }.executeQuery();
    }

}
