/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.groupware.userconfiguration;

import java.util.Map;
import java.util.Set;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.osgi.annotation.SingletonService;

/**
 * {@link PermissionConfigurationChecker} is a service that helps to check that no permissions are defined as capabilities and also creates
 * prominent log messages for them.
 * <p>
 * This service supports checks for user attributes, capabilities and for the configuration in general.
 *
 * @author <a href="mailto:kevin.ruthmann@open-xchange.com">Kevin Ruthmann</a>
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.10.4
 */
@SingletonService
public interface PermissionConfigurationChecker {

    /**
     * The property name for the setting whether it is allowed to perform provisioning operation with illegal capabilities.
     * <p>
     * A capability is rated as illegal if it denotes a permission; e.g. <code>"com.openexchange.capability.infostore"</code>.
     */
    public static final String PROP_ALLOW_ILLEGAL_PROV = "com.openexchange.capabilities.allowIllegalPermissionProvisioning";

    /**
     * The property name for the setting whether it is allowed to apply illegal capabilities.
     * <p>
     * A capability is rated as illegal if it denotes a permission; e.g. <code>"com.openexchange.capability.infostore"</code>.
     */
    public static final String PROP_APPLY_ILLEGAL_PERMISSIONS = "com.openexchange.capabilities.applyIllegalPermissions";

    // -------------------------------------------------------------------------------------------------------------------------------------

    /**
     * Checks the configuration if it contains any capability setting that denotes a permission; e.g. <code>"com.openexchange.capability.infostore"</code>.
     *
     * @param configService The service providing access to application's configuration
     */
    void checkConfig(ConfigurationService configService);

    /**
     * Checks whether the given attributes contains any capability setting that denotes a permission; e.g. <code>"com.openexchange.capability.infostore"</code>.
     *
     * @param attributes The user attributes to check
     * @throws OXException In case applying specified attributes is not allowed
     */
    void checkAttributes(Map<String, String> attributes) throws OXException;

    /**
     * Checks whether the given capability matches a permission, if so it is considered as illegal. E.g. <code>"infostore"</code>.
     *
     * @param capability The capability to check
     * @param userId The user identifier or <code>-1</code>
     * @param contextId The context identifier or <code>-1</code>
     * @return <code>true</code> if the permission is legal, <code>false</code> otherwise
     */
    boolean isLegal(String capability, int userId, int contextId);

    /**
     * Checks whether the given set contains any capability that matches a permission, if so it is considered as illegal. E.g. <code>"infostore"</code>.
     *
     * @param caps The capabilities to check
     * @throws OXException In case the set of capabilities is considered as illegal
     */
    void checkCapabilities(Set<String> caps) throws OXException;

}
