/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.external.accounts.clt;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import com.openexchange.auth.rmi.RemoteAuthenticator;
import com.openexchange.cli.AbstractRmiCLI;
import com.openexchange.external.account.ExternalAccountModule;
import com.openexchange.java.Enums;

/**
 * {@link AbstractExternalAccountCLT}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 * @since v7.10.4
 */
abstract class AbstractExternalAccountCLT extends AbstractRmiCLI<Void> {

    static final String USAGE = "[-c <contextId>] [-u <userId>] [-m <module>]";

    private final String footer;
    private final String syntax;

    /**
     * Initializes a new {@link AbstractExternalAccountCLT}.
     */
    public AbstractExternalAccountCLT(String syntax, String footer) {
        super();
        this.footer = footer;
        this.syntax = syntax;
    }

    @Override
    protected void addOptions(Options options) {
        options.addOption(createArgumentOption("c", "context", "contextId", "Required. The context identifier", true));
        options.addOption(createArgumentOption("u", "user", "userId", (isUserMandatory() ? "Required. " : "") + "The user identifier", isUserMandatory()));
        options.addOption(createArgumentOption("m", "module", "module", (isModuleMandatory() ? "Required. " : "") + "The module identifier. Valid modules are: " + getAvailableModules(), isModuleMandatory()));
    }

    @Override
    protected Boolean requiresAdministrativePermission() {
        return Boolean.TRUE;
    }

    @Override
    protected void administrativeAuth(String login, String password, CommandLine cmd, RemoteAuthenticator authenticator) throws Exception {
        authenticator.doAuthentication(login, password);
    }

    @Override
    protected String getFooter() {
        return footer;
    }

    @Override
    protected String getName() {
        return syntax;
    }

    /**
     * Gets the {@link ExternalAccountModule} from the command line arguments
     *
     * @param cmd the command line containing the arguments
     * @return The context identifier
     */
    ExternalAccountModule getModule(CommandLine cmd) {
        String module = cmd.getOptionValue('m');
        try {
            return ExternalAccountModule.valueOf(module.toUpperCase());
        } catch (IllegalArgumentException e) {
            System.err.println("Invalid value '" + module + "' for option 'm'. Available modules are: " + getAvailableModules());
            System.exit(1);
            return null;
        }
    }

    /**
     * Returns a comma-separated list with all available modules
     *
     * @return the string
     */
    String getAvailableModules() {
        return Enums.toCommaSeparatedList(ExternalAccountModule.values());
    }

    /**
     * Returns whether the user option is mandatory
     *
     * @return whether the user option is mandatory
     */
    abstract boolean isUserMandatory();

    /**
     * Returns whether the module option is mandatory
     *
     * @return whether the module option is mandatory
     */
    abstract boolean isModuleMandatory();
}
