/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.session;

import java.util.Arrays;
import java.util.EnumSet;
import com.openexchange.java.Strings;

/**
 * {@link Origin} - An enumeration for the origin of a spawned session.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.10.2
 */
public enum Origin {

    /**
     * HTTP/JSON channel was used to create the session.
     */
    HTTP_JSON,
    /**
     * The WebDAV access to Drive/InfoStore was used to create the session.
     */
    WEBDAV_INFOSTORE,
    /**
     * The WebDAV iCal access was used to create the session.
     */
    WEBDAV_ICAL,
    /**
     * The WebDAV vCard access was used to create the session.
     */
    WEBDAV_VCARD,
    /**
     * The Outlook Updater access was used to create the session.
     */
    OUTLOOK_UPDATER,
    /**
     * The Drive Updater access was used to create the session.
     */
    DRIVE_UPDATER,
    /**
     * The CalDAV access was used to create the session.
     */
    CALDAV,
    /**
     * The CardDAV access was used to create the session.
     */
    CARDDAV,
    /**
     * A synthetic session generated by the application.
     */
    SYNTHETIC;

    /**
     * Gets the origin for specified identifier
     *
     * @param origin The identifier
     * @return The origin or <code>null</code>
     */
    public static Origin originFor(String origin) {
        if (Strings.isEmpty(origin)) {
            return null;
        }

        String lookUp = Strings.toUpperCase(origin);
        for (Origin o : Origin.values()) {
            if (lookUp.equals(o.name())) {
                return o;
            }
        }
        return null;
    }

    /**
     * Checks if specified session is allowed to act in context of specified origins.
     *
     * @param session The session
     * @param origins The allowed origins
     * @return <code>true</code> if allowed; otherwise <code>false</code>
     */
    public static boolean isAllowed(Session session, Origin... origins ) {
        return (null == session || null == session.getOrigin() || null == origins || 0 == origins.length) || EnumSet.copyOf(Arrays.asList(origins)).contains(session.getOrigin());
    }

}
