package org.apache.jcs.engine.memory.lru;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jcs.engine.CacheConstants;
import org.apache.jcs.engine.behavior.ICacheElement;
import org.apache.jcs.engine.control.CompositeCache;
import org.apache.jcs.engine.control.group.GroupAttrName;
import org.apache.jcs.engine.control.group.GroupId;
import org.apache.jcs.engine.memory.AbstractMemoryCache;
import org.apache.jcs.engine.stats.StatElement;
import org.apache.jcs.engine.stats.Stats;
import org.apache.jcs.engine.stats.behavior.IStatElement;
import org.apache.jcs.engine.stats.behavior.IStats;

/**
 * A fast reference management system. The least recently used items move to the end of the list and
 * get spooled to disk if the cache hub is configured to use a disk cache. Most of the cache
 * bottelnecks are in IO. There are no io bottlenecks here, it's all about processing power.
 * <p>
 * Even though there are only a few adjustments necessary to maintain the double linked list, we
 * might want to find a more efficient memory manager for large cache regions.
 * <p>
 * The LRUMemoryCache is most efficient when the first element is selected. The smaller the region,
 * the better the chance that this will be the case. < .04 ms per put, p3 866, 1/10 of that per get
 * <p>
 * @version $Id: LRUMemoryCache.java 536904 2007-05-10 16:03:42Z tv $
 */
public class LRUMemoryCache
    extends AbstractMemoryCache
{
    /** Don't change */
    private static final long serialVersionUID = 6403738094136424201L;

    /** The logger. */
    private final static Log log = LogFactory.getLog( LRUMemoryCache.class );

    private static final Comparator<ICacheElement> NEWEST_FIRST_CACHE_ELEMENT_COMPARATOR = new Comparator<ICacheElement>() {

        @Override
        public int compare(ICacheElement o1, ICacheElement o2) {
            long x = o1.getElementAttributes().getLastAccessTime();
            long y = o2.getElementAttributes().getLastAccessTime();
            return x < y ? 1 : (x == y ? 0 : -1);
        }
    };

    private static class ShrinkerTask implements Runnable {

        private final ConcurrentMap<Serializable, ICacheElement> map;
        private final int numToSpool;
        private final LRUMemoryCache lruCache;

        ShrinkerTask(int numToSpool, ConcurrentMap<Serializable, ICacheElement> map, LRUMemoryCache lruCache) {
            super();
            this.map = map;
            this.numToSpool = numToSpool;
            this.lruCache = lruCache;
        }

        @Override
        public void run() {
            try {
                do {
                    lruCache.spoolLastElements(numToSpool);
                } while (map.size() > lruCache.cattr.getMaxObjects());
            } finally {
                lruCache.shrinkerFlag.set(false);
            }
        }
    }

    /** thread-safe double linked list for lru */
    //private ConcurrentLinkedDeque<ICacheElement> list;

    /** number of hits */
    private final java.util.concurrent.atomic.AtomicInteger hitCnt = new java.util.concurrent.atomic.AtomicInteger(0);

    /** number of misses */
    private final java.util.concurrent.atomic.AtomicInteger missCnt = new java.util.concurrent.atomic.AtomicInteger(0);

    /** number of puts */
    private final java.util.concurrent.atomic.AtomicInteger putCnt = new java.util.concurrent.atomic.AtomicInteger(0);

    /** current shrinker task */
    final java.util.concurrent.atomic.AtomicBoolean shrinkerFlag = new java.util.concurrent.atomic.AtomicBoolean(false);

    /**
     * For post reflection creation initialization.
     * <p>
     * @param hub
     */
    @Override
    public synchronized void initialize( CompositeCache hub )
    {
        super.initialize( hub );
        int maxObjects = this.cattr.getMaxObjects();
        int defaultChunkSize = chunkSize;
        chunkSize = Math.max(defaultChunkSize, ((int) (maxObjects * 0.20d)));
        // list = new ConcurrentLinkedDeque<ICacheElement>();
        log.info( "initialized LRUMemoryCache for " + cacheName );
    }

    /**
     * Puts an item to the cache. Removes any pre-existing entries of the same key from the linked
     * list and adds this one first.
     * <p>
     * If the max size is reached, an element will be put to disk.
     * <p>
     * @param ce The cache element, or entry wrapper
     * @exception IOException
     */
    @Override
    public void update( ICacheElement ce )
        throws IOException
    {
        putCnt.incrementAndGet();

        // Asynchronously create a MemoryElement

        ce.getElementAttributes().setLastAccessTimeNow();
        map.put( ce.getKey(), ce );

        // If the element limit is reached, we need to spool
        int size = map.size();
        if ( size <= this.cattr.getMaxObjects() )
        {
            return;
        }

        // Write the last 'chunkSize' items to disk.
        int chunkSizeCorrected = Math.min( size, chunkSize );
        shrinkOnPutIntoCache(chunkSizeCorrected);
    }

    private void shrinkOnPutIntoCache(final int numToSpool) {
        final AtomicBoolean shrinkerFlag = this.shrinkerFlag;
        if (shrinkerFlag.compareAndSet(false, true))
        {
            // Not yet running
            Runnable shrinkerTask = new ShrinkerTask(numToSpool, map, this);
            new Thread(shrinkerTask, this.cattr.getCacheName() + "-LRUMemoryCache-Shrinker").start();
        }
    }

    /**
     * This spools the last elements in the LRU, if some exist.
     * <p>
     * @param numToSpool The number of elements to spool
     * @return The elements that were spooled.
     * @throws Error
     */
    protected List<ICacheElement> spoolLastElements(int numToSpool)
        throws Error
    {
        List<ICacheElement> spooled = new ArrayList<>(numToSpool);
        int toAdd = numToSpool;
        for (ICacheElement ce : map.values())
        {
            if (toAdd > 0)
            {
                toAdd--;
                spooled.add(ce);
                if (spooled.size() > 1)
                {
                    Collections.sort(spooled, NEWEST_FIRST_CACHE_ELEMENT_COMPARATOR);
                }
            }
            else
            {
                if (ce.getElementAttributes().getLastAccessTime() < spooled.get(0).getElementAttributes().getLastAccessTime())
                {
                    spooled.set(0, ce);
                    if (spooled.size() > 1) {
                        Collections.sort(spooled, NEWEST_FIRST_CACHE_ELEMENT_COMPARATOR);
                    }
                }
            }
        }

        for (ICacheElement toSpool : spooled)
        {
            spoolElement(toSpool);
        }
        return spooled;
    }

    private void spoolElement(ICacheElement toSpool)
    {
        cache.spoolToDisk( toSpool );
        map.remove(toSpool.getKey());
    }

    /**
     * This instructs the memory cache to remove the <i>numberToFree</i> according to its eviction
     * policy. For example, the LRUMemoryCache will remove the <i>numberToFree</i> least recently
     * used items. These will be spooled to disk if a disk auxiliary is available.
     * <p>
     * @param numberToFree
     * @return the number that were removed. if you ask to free 5, but there are only 3, you will
     *         get 3.
     * @throws IOException
     */
    @Override
    public int freeElements( int numberToFree )
        throws IOException
    {
        return spoolLastElements(numberToFree).size();
    }

    /**
     * Get an item from the cache without affecting its last access time or position.
     * <p>
     * @param key Identifies item to find
     * @return Element matching key if found, or null
     * @exception IOException
     */
    @Override
    public ICacheElement getQuiet( Serializable key )
        throws IOException
    {
        return map.get( key );
    }

    /**
     * Get an item from the cache
     * <p>
     * @param key Identifies item to find
     * @return ICacheElement if found, else null
     * @exception IOException
     */
    @Override
    public ICacheElement get( Serializable key )
        throws IOException
    {
        ICacheElement ce = map.get( key );

        if ( ce != null )
        {
            hitCnt.incrementAndGet();

            ce.getElementAttributes().setLastAccessTimeNow();
        }
        else
        {
            missCnt.incrementAndGet();
        }

        return ce;
    }

    /**
     * Removes an item from the cache. This method handles hierarchical removal. If the key is a
     * String and ends with the CacheConstants.NAME_COMPONENT_DELIMITER, then all items with keys
     * starting with the argument String will be removed.
     * <p>
     * @param key
     * @return true if the removal was successful
     * @exception IOException
     */
    @Override
    public boolean remove( Serializable key )
        throws IOException
    {
        boolean removed = false;

        // handle partial removal
        if ( key instanceof String && ( (String) key ).endsWith( CacheConstants.NAME_COMPONENT_DELIMITER ) )
        {
            // remove all keys of the same name hierarchy.
            for ( Iterator<Serializable> itr = map.keySet().iterator(); itr.hasNext(); )
            {
                Object k = itr.next();

                if ( k instanceof String && ( (String) k ).startsWith( key.toString() ) )
                {
                    itr.remove();

                    removed = true;
                }
            }
        }
        else if ( key instanceof GroupId )
        {
            // remove all keys of the same name hierarchy.
            for ( Iterator<Serializable> itr = map.keySet().iterator(); itr.hasNext(); )
            {
                Object k = itr.next();

                if ( k instanceof GroupAttrName && ( (GroupAttrName) k ).groupId.equals( key ) )
                {
                    itr.remove();

                    removed = true;
                }
            }
        }
        else
        {
            // remove single item.
            ICacheElement ce = map.remove( key );

            if ( ce != null )
            {
                removed = true;
            }
        }

        return removed;
    }

    /**
     * Remove all of the elements from both the Map and the linked list implementation. Overrides
     * base class.
     * <p>
     * @throws IOException
     */
    @Override
    public void removeAll()
        throws IOException
    {
        map.clear();
    }

    // --------------------------- iteration mehods (iteration helpers)
    /**
     * iteration aid
     */
    public class IteratorWrapper
        implements Iterator<Map.Entry<Serializable, ICacheElement>>
    {
        // private final Log log = LogFactory.getLog( LRUMemoryCache.class );

        private final Iterator<Map.Entry<Serializable, ICacheElement>> i;

        IteratorWrapper( Map<Serializable, ICacheElement> m )
        {
            i = m.entrySet().iterator();
        }

        @Override
        public boolean hasNext()
        {
            return i.hasNext();
        }

        @Override
        public Map.Entry<Serializable, ICacheElement> next()
        {
            return new MapEntryWrapper( i.next() );
        }

        @Override
        public void remove()
        {
            i.remove();
        }

        @Override
        public boolean equals( Object o )
        {
            return i.equals( o );
        }

        @Override
        public int hashCode()
        {
            return i.hashCode();
        }
    }

    /**
     * @author Aaron Smuts
     */
    public class MapEntryWrapper
        implements Map.Entry<Serializable, ICacheElement>
    {
        private final Map.Entry<Serializable, ICacheElement> e;

        MapEntryWrapper( Map.Entry<Serializable, ICacheElement> e )
        {
            this.e = e;
        }

        @Override
        public boolean equals( Object o )
        {
            return e.equals( o );
        }

        @Override
        public Serializable getKey()
        {
            return e.getKey();
        }

        @Override
        public ICacheElement getValue()
        {
            return e.getValue();
        }

        @Override
        public int hashCode()
        {
            return e.hashCode();
        }

        @Override
        public ICacheElement setValue( ICacheElement value )
        {
            throw new UnsupportedOperationException( "Use normal cache methods"
                + " to alter the contents of the cache." );
        }
    }

    /**
     * Gets the iterator attribute of the LRUMemoryCache object
     * <p>
     * @return The iterator value
     */
    @Override
    public Iterator<Map.Entry<Serializable, ICacheElement>> getIterator()
    {
        return new IteratorWrapper( map );
    }

    /**
     * Get an Array of the keys for all elements in the memory cache
     * @return An Object[]
     */
    @Override
    public Object[] getKeyArray()
    {
        return map.keySet().toArray();
    }

    // ---------------------------------------------------------- debug methods

    /**
     * Dump the cache map for debugging.
     */
    public void dumpMap()
    {
        log.debug( "dumpingMap" );
        for ( Iterator<Map.Entry<Serializable, ICacheElement>> itr = map.entrySet().iterator(); itr.hasNext(); )
        {
            Map.Entry<Serializable, ICacheElement> e = itr.next();
            ICacheElement ce = e.getValue();
            log.debug( "dumpMap> key=" + e.getKey() + ", val=" + ce.getVal() );
        }
    }

    /**
     * Dump the cache entries from first to list for debugging.
     */
    public void dumpCacheEntries()
    {
        log.debug( "dumpingCacheEntries" );
        List<ICacheElement> values = new ArrayList<>(map.values());
        Collections.sort(values, NEWEST_FIRST_CACHE_ELEMENT_COMPARATOR);

        for ( ICacheElement ce : values )
        {
            log.debug( "dumpCacheEntries> key=" + ce.getKey() + ", val=" + ce.getVal() );
        }
    }

    /**
     * This returns semi-structured information on the memory cache, such as the size, put count,
     * hit count, and miss count.
     * <p>
     * @see org.apache.jcs.engine.memory.MemoryCache#getStatistics()
     */
    @Override
    public IStats getStatistics()
    {
        IStats stats = new Stats();
        stats.setTypeName( "LRU Memory Cache" );

        ArrayList<IStatElement> elems = new ArrayList<>();

        IStatElement se = null;

        /*-
         *
        se = new StatElement();
        se.setName( "List Size" );
        se.setData( "" + list.size() );
        elems.add( se );
        */

        se = new StatElement();
        se.setName( "Map Size" );
        se.setData( "" + map.size() );
        elems.add( se );

        se = new StatElement();
        se.setName( "Put Count" );
        se.setData( "" + putCnt );
        elems.add( se );

        se = new StatElement();
        se.setName( "Hit Count" );
        se.setData( "" + hitCnt );
        elems.add( se );

        se = new StatElement();
        se.setName( "Miss Count" );
        se.setData( "" + missCnt );
        elems.add( se );

        // get an array and put them in the Stats object
        IStatElement[] ses = elems.toArray( new StatElement[0] );
        stats.setStatElements( ses );

        // int rate = ((hitCnt + missCnt) * 100) / (hitCnt * 100) * 100;
        // buf.append("\n Hit Rate = " + rate + " %" );

        return stats;
    }

}
