/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.secret.recovery.json.action;

import java.util.Set;
import org.json.JSONException;
import org.slf4j.Logger;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.crypto.CryptoErrorMessage;
import com.openexchange.exception.OXException;
import com.openexchange.secret.SecretService;
import com.openexchange.secret.SecretUsesPasswordChecker;
import com.openexchange.secret.recovery.SecretMigrator;
import com.openexchange.secret.recovery.json.SecretRecoveryAJAXRequest;
import com.openexchange.server.ServiceExceptionCode;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.SetableSession;
import com.openexchange.session.SetableSessionFactory;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link MigrateAction}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public final class MigrateAction extends AbstractSecretRecoveryAction {

    /** Simple class to delay initialization until needed */
    private static class LoggerHolder {
        static final Logger LOG = org.slf4j.LoggerFactory.getLogger(MigrateAction.class);
    }

    private final Set<SecretMigrator> secretMigrators;

    /**
     * Initializes a new {@link MigrateAction}.
     *
     * @param services
     * @param secretMigrators
     */
    public MigrateAction(final ServiceLookup services, Set<SecretMigrator> secretMigrators) {
        super(services);
        this.secretMigrators = secretMigrators;
    }

    @Override
    protected AJAXRequestResult perform(final SecretRecoveryAJAXRequest req) throws OXException, JSONException {
        final SecretService secretService = getService(SecretService.class);
        if (null == secretService) {
            throw ServiceExceptionCode.SERVICE_UNAVAILABLE.create(SecretService.class.getName());
        }

        final String password = req.getParameter("password");
        final ServerSession session = req.getSession();

        // Old secret
        String oldSecret;
        SecretUsesPasswordChecker usesPasswordChecker = services.getOptionalService(SecretUsesPasswordChecker.class);
        if (usesPasswordChecker != null && usesPasswordChecker.usesPassword()) {
            SetableSession setableSession = SetableSessionFactory.getFactory().setableSessionFor(session);
            setableSession.setPassword(password);
            oldSecret = secretService.getSecret(setableSession);
        } else {
            oldSecret = password;
        }

        // New secret
        final String secret = secretService.getSecret(session);
        if (oldSecret.equals(secret)) {
            // Nothing to do...
            return new AJAXRequestResult(Integer.valueOf(1), "int");
        }

        for (SecretMigrator migrator : secretMigrators) {
            try {
                migrator.migrate(oldSecret, secret, session);
            } catch (OXException e) {
                if (!CryptoErrorMessage.BadPassword.equals(e)) {
                    throw e;
                }
                LoggerHolder.LOG.warn("{} is unable to re-crypt.", migrator.getClass().getName(), e);
            }
        }

        return new AJAXRequestResult(Integer.valueOf(1), "int");
    }

}
