/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.rest.client.httpclient.internal;

import com.openexchange.metrics.MetricDescriptor;
import com.openexchange.metrics.MetricDescriptor.MetricBuilder;
import com.openexchange.metrics.MetricType;

/**
 * {@link MonitoringId}
 *
 * @author <a href="mailto:steffen.templin@open-xchange.com">Steffen Templin</a>
 * @since v7.10.3
 */
public class MonitoringId {

    private static final MonitoringId NOOP = new MonitoringId("noop", 1);

    /**
     * Gets the special no-op instance.
     *
     * @return The no-op instance
     */
    public static MonitoringId getNoop() {
        return NOOP;
    }

    // -------------------------------------------------------------------------------------------------------------------------------------

    private final String clientName;

    private final int instanceId;

    /**
     * Initializes a new {@link MonitoringId}.
     *
     * @param clientName The client name
     * @param instanceId The instance identifier
     */
    public MonitoringId(String clientName, int instanceId) {
        super();
        this.clientName = clientName;
        this.instanceId = instanceId;
    }

    public String getClientName() {
        return clientName;
    }

    public int getInstanceId() {
        return instanceId;
    }

    public void applyDimensionsTo(MetricBuilder builder) {
        builder.addDimension("client", clientName);
        builder.addDimension("instance", Integer.toString(instanceId));
    }

    public MetricBuilder newMetricBuilder(String group, String name, MetricType type) {
        MetricBuilder builder = MetricDescriptor.newBuilder(group, name, type);
        applyDimensionsTo(builder);
        return builder;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((clientName == null) ? 0 : clientName.hashCode());
        result = prime * result + instanceId;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        MonitoringId other = (MonitoringId) obj;
        if (clientName == null) {
            if (other.clientName != null) {
                return false;
            }
        } else if (!clientName.equals(other.clientName)) {
            return false;
        }
        if (instanceId != other.instanceId) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return clientName + ':' + Integer.toString(instanceId);
    }

}
