/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.groupware.update.tools.console;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import com.openexchange.groupware.update.UpdateTaskService;

/**
 * {@link UpdateTaskForceCLT} - Command-Line access to reset version via update task toolkit.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public final class UpdateTaskForceCLT extends AbstractUpdateTasksCLT<Void> {

    /**
     * Entry point
     * 
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        new UpdateTaskForceCLT().execute(args);
    }

    private String schemaName;
    private String className;
    private int contextId = -1;

    private static final String FOOTER = "Force (re-)run of update task denoted by given class name on a specific schema or on all schemata or on a specific context.";

    /**
     * Initializes a new {@link UpdateTaskForceCLT}.
     */
    private UpdateTaskForceCLT() {
        super("forceupdatetask -t <taskName> [-c <contextId> | -n <schemaName>] " + BASIC_MASTER_ADMIN_USAGE, FOOTER);
    }

    @Override
    protected void addOptions(Options options) {
        options.addOption(createArgumentOption("t", "task", "taskName", "The update task's class name", false));

        StringBuilder sb = new StringBuilder(128);
        sb.append("A valid context identifier contained in target schema;");
        sb.append(" if missing and '-n/--name' option is also absent all schemas are considered.");
        options.addOption(createArgumentOption("c", "context", "contextId", sb.toString(), false));

        sb.setLength(0);
        sb.append("A valid schema name. This option is a substitute for '-c/--context' option.");
        sb.append(" If both are present '-c/--context' is preferred. If both absent all schemas are considered.");
        options.addOption(createArgumentOption("n", "name", "schemaName", sb.toString(), false));
    }

    @Override
    protected Void invoke(Options options, CommandLine cmd, String optRmiHostName) throws Exception {
        UpdateTaskService updateTaskService = getRmiStub(UpdateTaskService.RMI_NAME);
        boolean noName = (null == schemaName);
        if (noName && (-1 == contextId)) {
            updateTaskService.forceUpdateTaskOnAllSchemata(className);
            return null;
        }
        if (noName) {
            updateTaskService.forceUpdateTask(contextId, className);
            return null;
        }
        updateTaskService.forceUpdateTask(schemaName, className);
        return null;
    }

    @Override
    protected void checkOptions(CommandLine cmd) {
        if (!cmd.hasOption('t')) {
            System.err.println("Missing update task's class name.");
            printHelp();
            System.exit(1);
        } else {
            className = cmd.getOptionValue('t');
        }
        if (cmd.hasOption('c')) {
            String optionValue = cmd.getOptionValue('c');
            try {
                contextId = Integer.parseInt(optionValue.trim());
            } catch (NumberFormatException e) {
                System.err.println("Context identifier parameter is not a number: " + optionValue);
                printHelp();
                System.exit(1);
            }
        } else {
            if (cmd.hasOption('n')) {
                schemaName = cmd.getOptionValue('n');
            }
        }
    }
}
