/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mail.compose.json.converter;

import java.util.Collection;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.Converter;
import com.openexchange.ajax.requesthandler.ResultConverter;
import com.openexchange.exception.OXException;
import com.openexchange.java.util.UUIDs;
import com.openexchange.mail.compose.Attachment;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link AttachmentJSONResultConverter}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.10.2
 */
public class AttachmentJSONResultConverter implements ResultConverter {

    /**
     * Initializes a new {@link AttachmentJSONResultConverter}.
     */
    public AttachmentJSONResultConverter() {
        super();
    }

    @Override
    public String getInputFormat() {
        return "compositionSpaceAttachment";
    }

    @Override
    public String getOutputFormat() {
        return "json";
    }

    @Override
    public Quality getQuality() {
        return Quality.GOOD;
    }

    @Override
    public void convert(final AJAXRequestData requestData, final AJAXRequestResult result, final ServerSession session, final Converter converter) throws OXException {
        try {
            final Object resultObject = result.getResultObject();
            if (resultObject instanceof Attachment) {
                Attachment attachment = (Attachment) resultObject;
                result.setResultObject(convertAttachment(attachment), "json");
                return;
            }
            /*
             * Collection of attachments
             */
            @SuppressWarnings("unchecked")
            Collection<Attachment> attachments = (Collection<Attachment>) resultObject;
            final JSONArray jArray = new JSONArray(attachments.size());
            for (Attachment attachment : attachments) {
                jArray.put(convertAttachment(attachment));
            }
            result.setResultObject(jArray, "json");
        } catch (JSONException e) {
            throw AjaxExceptionCodes.JSON_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Converts specified attachment to its JSON representation.
     *
     * @param attachment The attachment
     * @return The JSON representation
     * @throws JSONException If conversion fails
     */
    public static JSONObject convertAttachment(Attachment attachment) throws JSONException {
        if (null == attachment) {
            return null;
        }

        JSONObject jAttachment = new JSONObject(8);
        jAttachment.putOpt("id", UUIDs.getUnformattedString(attachment.getId()));
        jAttachment.putOpt("name", attachment.getName());
        jAttachment.put("size", attachment.getSize());
        jAttachment.putOpt("mimeType", attachment.getMimeType());
        jAttachment.putOpt("cid", attachment.getContentId());
        jAttachment.putOpt("contentDisposition", attachment.getContentDisposition());
        jAttachment.putOpt("origin", attachment.getOrigin());
        return jAttachment;
    }

}
