/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.oauth;

import static com.openexchange.java.Strings.isEmpty;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import com.openexchange.exception.OXException;
import com.openexchange.framework.request.RequestContext;
import com.openexchange.framework.request.RequestContextHolder;
import com.openexchange.groupware.notify.hostname.HostData;
import com.openexchange.java.Strings;
import com.openexchange.oauth.scope.OAuthScope;
import com.openexchange.oauth.scope.OXScope;
import com.openexchange.session.Session;

/**
 * {@link OAuthUtil} - Utility class for OAuth.
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public final class OAuthUtil {

    /**
     * Checks that specified scopes are both - available and enabled.
     *
     * @param oauthAccount The OAuth account providing enabled scopes
     * @param session The session providing user data
     * @param scopes The scopes to check
     * @throws OXException If one of specified scopes is either not availabke or not enabled
     */
    public static void checkScopesAvailableAndEnabled(OAuthAccount oauthAccount, Session session, OXScope... scopes) throws OXException {
        checkScopesAvailableAndEnabled(oauthAccount, session.getUserId(), session.getContextId(), scopes);
    }

    /**
     * Checks that specified scopes are both - available and enabled.
     *
     * @param oauthAccount The OAuth account providing enabled scopes
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param scopes The scopes to check
     * @throws OXException If one of specified scopes is either not availabke or not enabled
     */
    public static void checkScopesAvailableAndEnabled(OAuthAccount oauthAccount, int userId, int contextId, OXScope... scopes) throws OXException {
        if (null == oauthAccount) {
            return;
        }
        if (null == scopes || scopes.length <= 0) {
            return;
        }

        OAuthServiceMetaData oAuthServiceMetaData = oauthAccount.getMetaData();

        // Check the scopes permitted through "com.openexchange.oauth.modules.enabled + <service>" property
        {
            Set<OAuthScope> availableScopes = oAuthServiceMetaData.getAvailableScopes(userId, contextId);
            for (OXScope oxScope : scopes) {
                OAuthScope foundScope = null;
                for (Iterator<OAuthScope> it = availableScopes.iterator(); null == foundScope && it.hasNext();) {
                    OAuthScope scope = it.next();
                    if (oxScope == scope.getOXScope()) {
                        foundScope = scope;
                    }
                }
                if (null == foundScope) {
                    throw OAuthExceptionCodes.NO_SUCH_SCOPE_AVAILABLE.create(oxScope.getDisplayName());
                }
            }
        }

        // Check the scopes enabled for the account
        {
            Set<OAuthScope> enabledScopes = oauthAccount.getEnabledScopes();
            for (OXScope oxScope : scopes) {
                boolean supported = false;
                for (Iterator<OAuthScope> it = enabledScopes.iterator(); !supported && it.hasNext();) {
                    if (oxScope == it.next().getOXScope()) {
                        supported = true;
                    }
                }
                if (false == supported) {
                    throw OAuthExceptionCodes.NO_SCOPE_PERMISSION.create(oAuthServiceMetaData.getDisplayName(), oxScope.getDisplayName());
                }
            }
        }
    }

    /**
     * Parses the specified {@link Set} with {@link OAuthScope}s and returns
     * the OAuth provider-specific mappings ({@link OAuthScope#getProviderScopes()})
     * as a space separated string. Duplicate OAuth provider-specific scopes will only be
     * present once in the returned string.
     *
     * @param scopes The {@link OAuthScope}s
     * @return a space separated string with all {@link OAuthScope}s in the specified {@link Set}
     */
    public static final String providerScopesToString(Set<OAuthScope> scopes) {
        Set<String> strings = new HashSet<>();
        for (OAuthScope scope : scopes) {
            String[] split = Strings.splitByWhitespaces(scope.getProviderScopes());
            for (String s : split) {
                strings.add(s);
            }
        }
        return setToString(strings);
    }

    /**
     * Parses the specified {@link Set} with {@link OAuthScope}s and returns
     * the ({@link OAuthScope#getOXScope()}) as a whitespace separated string
     *
     * @param scopes The {@link OAuthScope}s
     * @return a space separated string with all {@link OAuthScope}s in the specified {@link Set}
     */
    public static final String oxScopesToString(Set<OAuthScope> scopes) {
        Set<String> strings = new HashSet<>();
        for (OAuthScope scope : scopes) {
            strings.add(scope.getOXScope().name());
        }
        return setToString(strings);
    }

    /**
     * Creates a whitespace separated list of strings out of the specified {@link Set}
     *
     * @param strings The {@link Set} with the strings
     * @return a whitespace separated list of strings
     */
    private static final String setToString(Set<String> strings) {
        if (strings.isEmpty()) {
            return "";
        }
        StringBuilder builder = new StringBuilder();
        for (String string : strings) {
            builder.append(string).append(" ");
        }
        builder.setLength(builder.length() - 1);
        return builder.toString();
    }

    /**
     * Builds the 'init' call-back URL for the given {@link OAuthAccount}
     *
     * @param account The {@link OAuthAccount}
     *
     * @return the 'init' call-back URL for the given {@link OAuthAccount}
     */
    public static final String buildCallbackURL(OAuthAccount account) {
        RequestContext requestContext = RequestContextHolder.get();
        if (null == requestContext) {
            return null;
        }

        HostData hostData = requestContext.getHostData();
        boolean isSecure = hostData.isSecure();

        StringBuilder builder = new StringBuilder();
        builder.append(isSecure ? "https://" : "http://");
        builder.append(determineHost(hostData));
        builder.append(hostData.getDispatcherPrefix());
        builder.append("oauth/accounts?action=init");
        builder.append("&serviceId=").append(account.getAPI().getName());
        builder.append("&id=").append(account.getId());
        builder.append('&').append(OAuthConstants.ARGUMENT_DISPLAY_NAME).append('=').append(urlEncode(account.getDisplayName()));
        builder.append("&scopes=").append(urlEncode(OAuthUtil.oxScopesToString(account.getEnabledScopes())));

        return builder.toString();
    }

    /**
     * Tries to determine the hostname by first looking in to {@link HostData},
     * then through Java and if still not available, falls back to 'localhost' as last resort.
     *
     * @param hostData The {@link HostData}
     * @return The hostname
     */
    private static final String determineHost(HostData hostData) {
        // Try from the host data
        String hostname = hostData.getHost();

        // Get hostname from java
        if (isEmpty(hostname)) {
            try {
                hostname = InetAddress.getLocalHost().getCanonicalHostName();
            } catch (UnknownHostException e) {
                // ignore
            }
        }
        // Fall back to localhost as last resort
        if (isEmpty(hostname)) {
            hostname = "localhost";
        }

        return hostname;
    }

    /**
     * URL encodes the specified string using "ISO-8859-1"
     *
     * @param s The string to encode
     * @return The encoded string
     */
    private static String urlEncode(final String s) {
        try {
            return URLEncoder.encode(s, "ISO-8859-1");
        } catch (final UnsupportedEncodingException e) {
            return s;
        }
    }

    /**
     * Returns the OAuth account identifier from associated account's configuration
     *
     * @param configuration The configuration
     * @return The account identifier
     * @throws IllegalArgumentException If the configuration is <code>null</code>, or if the account identifier is not present, or is present but cannot be parsed as an integer
     */
    public static int getAccountId(Map<String, Object> configuration) {
        if (null == configuration) {
            throw new IllegalArgumentException("The configuration cannot be 'null'");
        }

        Object accountId = configuration.get("account");
        if (null == accountId) {
            throw new IllegalArgumentException("The account identifier is missing from the configuration");
        }

        if (accountId instanceof Integer) {
            return ((Integer) accountId).intValue();
        }

        try {
            return Integer.parseInt(accountId.toString());
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("The account identifier '" + accountId.toString() + "' cannot be parsed as an integer.", e);
        }
    }
}
