package org.apache.jcs.engine.memory.lru;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedDeque;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jcs.engine.CacheConstants;
import org.apache.jcs.engine.CacheElement;
import org.apache.jcs.engine.behavior.ICacheElement;
import org.apache.jcs.engine.control.CompositeCache;
import org.apache.jcs.engine.control.group.GroupAttrName;
import org.apache.jcs.engine.control.group.GroupId;
import org.apache.jcs.engine.memory.AbstractMemoryCache;
import org.apache.jcs.engine.stats.StatElement;
import org.apache.jcs.engine.stats.Stats;
import org.apache.jcs.engine.stats.behavior.IStatElement;
import org.apache.jcs.engine.stats.behavior.IStats;

/**
 * A fast reference management system. The least recently used items move to the end of the list and
 * get spooled to disk if the cache hub is configured to use a disk cache. Most of the cache
 * bottelnecks are in IO. There are no io bottlenecks here, it's all about processing power.
 * <p>
 * Even though there are only a few adjustments necessary to maintain the double linked list, we
 * might want to find a more efficient memory manager for large cache regions.
 * <p>
 * The LRUMemoryCache is most efficient when the first element is selected. The smaller the region,
 * the better the chance that this will be the case. < .04 ms per put, p3 866, 1/10 of that per get
 * <p>
 * @version $Id: LRUMemoryCache.java 536904 2007-05-10 16:03:42Z tv $
 */
public class LRUMemoryCache
    extends AbstractMemoryCache
{
    /** Don't change */
    private static final long serialVersionUID = 6403738094136424201L;

    /** The logger. */
    private final static Log log = LogFactory.getLog( LRUMemoryCache.class );

    /** thread-safe double linked list for lru */
    private ConcurrentLinkedDeque<ICacheElement> list;

    /** number of hits */
    private final java.util.concurrent.atomic.AtomicInteger hitCnt = new java.util.concurrent.atomic.AtomicInteger(0);

    /** number of misses */
    private final java.util.concurrent.atomic.AtomicInteger missCnt = new java.util.concurrent.atomic.AtomicInteger(0);

    /** number of puts */
    private final java.util.concurrent.atomic.AtomicInteger putCnt = new java.util.concurrent.atomic.AtomicInteger(0);

    /**
     * For post reflection creation initialization.
     * <p>
     * @param hub
     */
    @Override
    public synchronized void initialize( CompositeCache hub )
    {
        super.initialize( hub );
        list = new ConcurrentLinkedDeque<ICacheElement>();
        log.info( "initialized LRUMemoryCache for " + cacheName );
    }

    /**
     * Puts an item to the cache. Removes any pre-existing entries of the same key from the linked
     * list and adds this one first.
     * <p>
     * If the max size is reached, an element will be put to disk.
     * <p>
     * @param ce The cache element, or entry wrapper
     * @exception IOException
     */
    @Override
    public void update( ICacheElement ce )
        throws IOException
    {
        putCnt.incrementAndGet();

        // Asynchronously create a MemoryElement

        ce.getElementAttributes().setLastAccessTimeNow();
        ICacheElement old = null;
        ICacheElement first = addFirst( ce );
        old = (ICacheElement) map.put( first.getKey(), first );
        // If the node was the same as an existing node, remove it.
        if ( old != null && first.getKey().equals( old.getKey() ) )
        {
            list.remove( old );
        }

        int size = map.size();
        // If the element limit is reached, we need to spool

        if ( size < this.cattr.getMaxObjects() )
        {
            return;
        }

        // Write the last 'chunkSize' items to disk.
        int chunkSizeCorrected = Math.min( size, chunkSize );

        // The spool will put them in a disk event queue, so there is no
        // need to pre-queue the queuing. This would be a bit wasteful
        // and wouldn't save much time in this synchronous call.

        for ( int i = 0; i < chunkSizeCorrected; i++ )
        {
            spoolLastElement();
        }
    }

    /**
     * This spools the last element in the LRU, if one exists.
     * <p>
     * @return ICacheElement if there was a last element, else null.
     * @throws Error
     */
    protected ICacheElement spoolLastElement()
        throws Error
    {
        ICacheElement toSpool = list.peekLast();
        if ( toSpool != null )
        {
            cache.spoolToDisk( toSpool );
            if ( !map.containsKey( toSpool.getKey() ) )
            {
                log.error( "update: map does not contain key: " + toSpool.getKey() );
            }
            if ( map.remove( toSpool.getKey() ) == null )
            {
                log.warn( "update: remove failed for key: " + toSpool.getKey() );
            }
            list.removeFirstOccurrence(toSpool);
        }
        else
        {
            throw new Error( "update: last.ce is null!" );
        }

        // If this is out of the sync block it can detect a mismatch
        // where there is none.
        if ( map.size() != dumpCacheSize() )
        {
            log.warn( "update: After spool, size mismatch: map.size() = " + map.size() + ", linked list size = "
                + dumpCacheSize() );
        }
        return toSpool;
    }

    /**
     * This instructs the memory cache to remove the <i>numberToFree</i> according to its eviction
     * policy. For example, the LRUMemoryCache will remove the <i>numberToFree</i> least recently
     * used items. These will be spooled to disk if a disk auxiliary is available.
     * <p>
     * @param numberToFree
     * @return the number that were removed. if you ask to free 5, but there are only 3, you will
     *         get 3.
     * @throws IOException
     */
    @Override
    public int freeElements( int numberToFree )
        throws IOException
    {
        int freed = 0;
        for ( ; freed < numberToFree; freed++ )
        {
            ICacheElement element = spoolLastElement();
            if ( element == null )
            {
                break;
            }
        }
        return freed;
    }

    /**
     * Get an item from the cache without affecting its last access time or position.
     * <p>
     * @param key Identifies item to find
     * @return Element matching key if found, or null
     * @exception IOException
     */
    @Override
    public ICacheElement getQuiet( Serializable key )
        throws IOException
    {
        return (ICacheElement) map.get( key );
    }

    /**
     * Get an item from the cache
     * <p>
     * @param key Identifies item to find
     * @return ICacheElement if found, else null
     * @exception IOException
     */
    @Override
    public ICacheElement get( Serializable key )
        throws IOException
    {
        ICacheElement ce = (ICacheElement) map.get( key );

        if ( ce != null )
        {
            hitCnt.incrementAndGet();

            ce.getElementAttributes().setLastAccessTimeNow();
            if (ce != list.peekFirst()) {
                list.removeFirstOccurrence(ce);
                list.addFirst(ce);
            }
        }
        else
        {
            missCnt.incrementAndGet();
        }

        return ce;
    }

    /**
     * Removes an item from the cache. This method handles hierarchical removal. If the key is a
     * String and ends with the CacheConstants.NAME_COMPONENT_DELIMITER, then all items with keys
     * starting with the argument String will be removed.
     * <p>
     * @param key
     * @return true if the removal was successful
     * @exception IOException
     */
    @Override
    public boolean remove( Serializable key )
        throws IOException
    {
        boolean removed = false;

        // handle partial removal
        if ( key instanceof String && ( (String) key ).endsWith( CacheConstants.NAME_COMPONENT_DELIMITER ) )
        {
            // remove all keys of the same name hierarchy.
            for ( Iterator<Map.Entry<Serializable, Serializable>> itr = map.entrySet().iterator(); itr.hasNext(); )
            {
                Map.Entry<Serializable, Serializable> entry = itr.next();
                Object k = entry.getKey();

                if ( k instanceof String && ( (String) k ).startsWith( key.toString() ) )
                {
                    list.remove( entry.getValue() );

                    itr.remove();

                    removed = true;
                }
            }
        }
        else if ( key instanceof GroupId )
        {
            // remove all keys of the same name hierarchy.
            for ( Iterator<Map.Entry<Serializable, Serializable>> itr = map.entrySet().iterator(); itr.hasNext(); )
            {
                Map.Entry<Serializable, Serializable> entry = itr.next();
                Object k = entry.getKey();

                if ( k instanceof GroupAttrName && ( (GroupAttrName) k ).groupId.equals( key ) )
                {
                    itr.remove();

                    list.remove( entry.getValue() );

                    removed = true;
                }
            }
        }
        else
        {
            // remove single item.
            ICacheElement ce = (ICacheElement) map.remove( key );

            if ( ce != null )
            {
                list.remove( ce );
                removed = true;
            }
        }

        return removed;
    }

    /**
     * Remove all of the elements from both the Map and the linked list implementation. Overrides
     * base class.
     * <p>
     * @throws IOException
     */
    @Override
    public void removeAll()
        throws IOException
    {
        map.clear();
        list.clear();
    }

    // --------------------------- iteration mehods (iteration helpers)
    /**
     * iteration aid
     */
    public class IteratorWrapper
        implements Iterator<Map.Entry<Serializable, Serializable>>
    {
        // private final Log log = LogFactory.getLog( LRUMemoryCache.class );

        private final Iterator<Map.Entry<Serializable, Serializable>> i;

        IteratorWrapper( Map<Serializable, Serializable> m )
        {
            i = m.entrySet().iterator();
        }

        @Override
        public boolean hasNext()
        {
            return i.hasNext();
        }

        @Override
        public Map.Entry<Serializable, Serializable> next()
        {
            return new MapEntryWrapper( i.next() );
        }

        @Override
        public void remove()
        {
            i.remove();
        }

        @Override
        public boolean equals( Object o )
        {
            return i.equals( o );
        }

        @Override
        public int hashCode()
        {
            return i.hashCode();
        }
    }

    /**
     * @author Aaron Smuts
     */
    public class MapEntryWrapper
        implements Map.Entry<Serializable, Serializable>
    {
        private final Map.Entry<Serializable, Serializable> e;

        MapEntryWrapper( Map.Entry<Serializable, Serializable> e )
        {
            this.e = e;
        }

        @Override
        public boolean equals( Object o )
        {
            return e.equals( o );
        }

        @Override
        public Serializable getKey()
        {
            return e.getKey();
        }

        @Override
        public ICacheElement getValue()
        {
            return (ICacheElement) e.getValue();
        }

        @Override
        public int hashCode()
        {
            return e.hashCode();
        }

        @Override
        public Serializable setValue( Serializable value )
        {
            throw new UnsupportedOperationException( "Use normal cache methods"
                + " to alter the contents of the cache." );
        }
    }

    /**
     * Gets the iterator attribute of the LRUMemoryCache object
     * <p>
     * @return The iterator value
     */
    @Override
    public Iterator<Map.Entry<Serializable, Serializable>> getIterator()
    {
        return new IteratorWrapper( map );
    }

    /**
     * Get an Array of the keys for all elements in the memory cache
     * @return An Object[]
     */
    @Override
    public Object[] getKeyArray()
    {
        return map.keySet().toArray();
    }

    // --------------------------- internal methods (linked list implementation)
    /**
     * Adds a new node to the end of the link list. Currently not used.
     * <p>
     * @param ce The feature to be added to the Last
     */
    protected void addLast( CacheElement ce )
    {
        list.addLast( ce );
    }

    /**
     * Adds a new node to the start of the link list.
     * <p>
     * @param ce The feature to be added to the First
     */
    private ICacheElement addFirst( ICacheElement ce )
    {
        list.addFirst( ce );
        return ce;
    }

    // ---------------------------------------------------------- debug methods

    /**
     * Dump the cache map for debugging.
     */
    public void dumpMap()
    {
        log.debug( "dumpingMap" );
        for ( Iterator<Map.Entry<Serializable, Serializable>> itr = map.entrySet().iterator(); itr.hasNext(); )
        {
            Map.Entry<Serializable, Serializable> e = itr.next();
            ICacheElement ce = (ICacheElement) e.getValue();
            log.debug( "dumpMap> key=" + e.getKey() + ", val=" + ce.getVal() );
        }
    }

    /**
     * Dump the cache entries from first to list for debugging.
     */
    public void dumpCacheEntries()
    {
        log.debug( "dumpingCacheEntries" );
        for ( ICacheElement ce = list.pollFirst(); ce != null; ce = list.pollFirst() )
        {
            log.debug( "dumpCacheEntries> key=" + ce.getKey() + ", val=" + ce.getVal() );
        }
    }

    /**
     * Returns the size of the list.
     * <p>
     * @return the number of items in the map.
     */
    private int dumpCacheSize()
    {
        return list.size();
    }

    /**
     * This returns semi-structured information on the memory cache, such as the size, put count,
     * hit count, and miss count.
     * <p>
     * @see org.apache.jcs.engine.memory.MemoryCache#getStatistics()
     */
    @Override
    public IStats getStatistics()
    {
        IStats stats = new Stats();
        stats.setTypeName( "LRU Memory Cache" );

        ArrayList<IStatElement> elems = new ArrayList<>();

        IStatElement se = null;

        se = new StatElement();
        se.setName( "List Size" );
        se.setData( "" + list.size() );
        elems.add( se );

        se = new StatElement();
        se.setName( "Map Size" );
        se.setData( "" + map.size() );
        elems.add( se );

        se = new StatElement();
        se.setName( "Put Count" );
        se.setData( "" + putCnt );
        elems.add( se );

        se = new StatElement();
        se.setName( "Hit Count" );
        se.setData( "" + hitCnt );
        elems.add( se );

        se = new StatElement();
        se.setName( "Miss Count" );
        se.setData( "" + missCnt );
        elems.add( se );

        // get an array and put them in the Stats object
        IStatElement[] ses = elems.toArray( new StatElement[0] );
        stats.setStatElements( ses );

        // int rate = ((hitCnt + missCnt) * 100) / (hitCnt * 100) * 100;
        // buf.append("\n Hit Rate = " + rate + " %" );

        return stats;
    }
}
