/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.realtime.atmosphere.impl;

import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.atmosphere.cpr.AtmosphereResource;
import org.json.JSONArray;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.log.LogFactory;
import com.openexchange.realtime.atmosphere.impl.stanza.writer.StanzaWriter;
import com.openexchange.realtime.atmosphere.protocol.StanzaTransmitter;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Stanza;


/**
 * The {@link AtmosphereStanzaTransmitter} is a {@link StanzaTransmitter} that is based on an {@link AtmosphereResource}.
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class AtmosphereStanzaTransmitter implements StanzaTransmitter {
    
    private static final Log LOG = LogFactory.getLog(AtmosphereStanzaTransmitter.class);
    
    private AtmosphereResource resource;
    private Lock lock = new ReentrantLock();
    private boolean canSend = true;
    private StateManager stateManager;
    private ID id;
    
    public AtmosphereStanzaTransmitter(AtmosphereResource resource, ID id, StateManager stateManager) {
        super();
        this.resource = resource;
        this.stateManager = stateManager;
        this.id = id;
    }

    @Override
    public boolean send(List<Stanza> stanzas) throws OXException {
        if (stanzas.isEmpty()) {
            return true;
        }
        try {
            lock.lock();
            trace(stanzas, "Trying to send as part of a batch of " +stanzas.size() + " stanzas");
            if (!canSend || resource.isCancelled() || resource.isResumed() || stanzas.isEmpty()) {
                trace(stanzas, "Stale AtmosphereStanzaTransmitter, enqueue again");
                return false;
            }
            JSONArray array = new JSONArray();
            StanzaWriter stanzaWriter = new StanzaWriter();
            for (Stanza stanza : stanzas) {
                JSONObject serialized = stanzaWriter.write(stanza);
                serialized.remove("to"); // Redundant
                array.put(serialized);
            }
            trace(stanzas, "Writing to stream");
            resource.getResponse().write(array.toString());
            switch (resource.transport()) {
            case LONG_POLLING: 
                if (resource.isSuspended()) {
                    trace(stanzas, "Delivering to client");
                    resource.resume(); 
                }
                canSend = false;
                stateManager.forgetTransmitter(id, this);
            }
            trace(stanzas, "Done sending");
            return true;
        } catch (Throwable t) {
            return false;
        } finally {
            lock.unlock();
        }
    }

    private void trace(List<Stanza> stanzas, String string) {
        LOG.debug(string);
        for (Stanza stanza : stanzas) {
            stanza.trace(string);
        }
    }

    @Override
    public void suspend() {
        try {
            lock.lock();
            if (resource.isCancelled() || resource.isResumed() || resource.isSuspended()) {
                return;
            }
            resource.suspend();
        } finally {
            lock.unlock();
        }
    }

}
