/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.i18n.impl;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.util.Locale;

/**
 * @author Francisco Laguna <francisco.laguna@open-xchange.com>
 */
public class FileDiscoverer {

    private final File dir;

    /**
     * Initializes a new {@link FileDiscoverer}.
     *
     * @param dir The directory
     * @throws FileNotFoundException If directory could not be found
     */
    public FileDiscoverer(final File dir) throws FileNotFoundException {
        super();
        if (!dir.exists()) {
            throw new FileNotFoundException("Unable to load language files. Directory does not exist: " + dir);
        }
        if (dir.isFile()) {
            throw new FileNotFoundException("Unable to load language files." + dir + " is not a directory");
        }
        this.dir = dir;
    }

    /**
     * Gets the files with given file extension
     *
     * @param extension The file extension; e.g. <code>".po"</code>
     * @return The matching files
     */
    public String[] getFilesFromLanguageFolder(final String extension) {
        final String[] files = dir.list(new FilenameFilter() {

            @Override
            public boolean accept(final File d, final String f) {
                return f.endsWith(extension);
            }
        });
        return files;
    }

    /**
     * Parses the locale from given file name; e.g. <code>"backend.<b>en_US</b>.po"</code>.
     *
     * @param fileName The file name
     * @return The parsed locale or <code>null</code>
     */
    public Locale getLocale(final String fileName) {
        final int indexOfUnderscore = fileName.indexOf('_');
        if (indexOfUnderscore < 0) {
            return null;
        }
        final int indexOfLastDot = fileName.lastIndexOf('.');
        if (indexOfLastDot < indexOfUnderscore) {
            return null;
        }
        final int indexOfDotBeforeUnderscore = fileName.lastIndexOf('.', indexOfUnderscore);

        if (indexOfUnderscore != -1) {
            final String language = fileName.substring(indexOfDotBeforeUnderscore + 1, indexOfUnderscore);
            final String country = fileName.substring(indexOfUnderscore + 1, indexOfLastDot);
            return new Locale(language, country);
        }
        return null;
    }

    /**
     * Gets the directory.
     *
     * @return The directory
     */
    public File getDirectory() {
        return dir;
    }
}
