/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.api;

import static com.openexchange.obs.api.Tools.transform;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * {@link ProjectParser}
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class ProjectParser {

    private static final DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

    private ProjectParser() {
        super();
    }

    public static Project parse(final InputStream is) throws BuildServiceException {
        final Document document;
        try {
            final DocumentBuilder db = dbf.newDocumentBuilder();
            document = db.parse(is);
            final NodeList childs = document.getChildNodes();
            for (int i = 0; i < childs.getLength(); i++) {
                final Node node = childs.item(i);
                if ("project".equals(node.getNodeName())) {
                     return parse(node);
                } else if (Node.TEXT_NODE == node.getNodeType()) {
                    continue;
                } else {
                    throw new BuildServiceException("Unknown node in project meta \"" + node.getNodeName() + "\".");
                }
            }
        } catch (ParserConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (SAXException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
        throw new BuildServiceException("Unable to parse project meta. " + transform(document));
    }

    public static String getDisabled(final InputStream is) throws BuildServiceException {
        final Document document;
        try {
            final DocumentBuilder db = dbf.newDocumentBuilder();
            document = db.parse(is);
            final NodeList childs = document.getChildNodes();
            for (int i = 0; i < childs.getLength(); i++) {
                final Node node = childs.item(i);
                if ("project".equals(node.getNodeName())) {
                    disable(node, document);
                }
            }
        } catch (ParserConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (SAXException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
        return transform(document);
    }

    public static boolean isDisabled(final InputStream is) throws BuildServiceException {
        final Document document;
        try {
            final DocumentBuilder db = dbf.newDocumentBuilder();
            document = db.parse(is);
            final NodeList childs = document.getChildNodes();
            for (int i = 0; i < childs.getLength(); i++) {
                final Node node = childs.item(i);
                if ("project".equals(node.getNodeName())) {
                    return isDisabled(node);
                }
            }
            return false;
        } catch (ParserConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (SAXException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    private static boolean isDisabled(final Node project) throws DOMException {
        final NodeList childs = project.getChildNodes();
        boolean builddisabled = false;
        boolean publishdisabled = false;
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("build".equals(node.getNodeName())) {
                builddisabled = true;
            }
            if ("publish".equals(node.getNodeName())) {
                publishdisabled = true;
            }
        }
        return builddisabled && publishdisabled;
    }

    private static void disable(final Node project, final Document document) throws DOMException {
        final NodeList childs = project.getChildNodes();
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("build".equals(node.getNodeName())) {
                project.removeChild(node);
            }
            if ("publish".equals(node.getNodeName())) {
                project.removeChild(node);
            }
        }
        final Element buildElement = document.createElement("build");
        final Element publishElement = document.createElement("publish");
        buildElement.appendChild(document.createElement("disable"));
        publishElement.appendChild(document.createElement("disable"));
        project.appendChild(buildElement);
        project.appendChild(publishElement);
    }

    public static Project[] parseAll(final InputStream is) throws BuildServiceException {
        final Document document;
        try {
            final DocumentBuilder db = dbf.newDocumentBuilder();
            document = db.parse(is);
            final NodeList childs = document.getChildNodes();
            for (int i = 0; i < childs.getLength(); i++) {
                final Node node = childs.item(i);
                if ("directory".equals(node.getNodeName())) {
                    return parseProjectlist(node);
                } else if (Node.TEXT_NODE == node.getNodeType()) {
                    continue;
                } else {
                    throw new BuildServiceException("Unknown node in project meta \"" + node.getNodeName() + "\".");
                }
            }
        } catch (ParserConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (SAXException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
        throw new BuildServiceException("Unable to parse project meta. " + transform(document));
    }

    private static Project[] parseProjectlist(final Node projectList) throws DOMException, BuildServiceException {
        final NodeList childs = projectList.getChildNodes();
        final List<Project> results = new ArrayList<Project>();
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("entry".equals(node.getNodeName())) {
                results.add(parse(node));
            } else if (Node.TEXT_NODE == node.getNodeType()) {
                continue;
            } else {
                throw new BuildServiceException("Unknown node in project list \"" + node.getNodeName() + "\" \"" + node.getNodeValue() + "\".");
            }
        }
        return results.toArray(new Project[results.size()]);
    }

    private static Project parse(final Node node) throws BuildServiceException {
        final NamedNodeMap attributes = node.getAttributes();
        final String name = attributes.getNamedItem("name").getNodeValue();
        final Project project = new Project(name);
        for (Repository repository : parseProject(node)) {
            project.addRepository(repository);
        }
        return project;
    }

    private static Repository[] parseProject(final Node projectNode) throws BuildServiceException {
        final NodeList childs = projectNode.getChildNodes();
        final List<Repository> packages = new ArrayList<Repository>();
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("repository".equals(node.getNodeName())) {
                packages.add(parseRepository(node));
            } else if (Node.TEXT_NODE == node.getNodeType()) {
                continue;
            } else if ("title".equals(node.getNodeName())) {
                continue;
            } else if ("description".equals(node.getNodeName())) {
                continue;
            } else if ("person".equals(node.getNodeName())) {
                continue;
            } else if ("build".equals(node.getNodeName())) {
                continue;
            } else if ("publish".equals(node.getNodeName())) {
                continue;
            } else {
                throw new BuildServiceException("Unknown node in result \"" + node.getNodeName() + "\" \"" + node.getNodeValue() + "\".");
            }
        }
        return packages.toArray(new Repository[packages.size()]);
    }

    private static Repository parseRepository(final Node repositoryNode) throws BuildServiceException {
        final NamedNodeMap attributes = repositoryNode.getAttributes();
        String repositoryName = attributes.getNamedItem("name").getNodeValue();
        String arch = null;
        final NodeList childs = repositoryNode.getChildNodes();
        for (int i = 0; i < childs.getLength(); i++) {
            Node node = childs.item(i);
            if ("arch".equals(node.getNodeName())) {
                arch = node.getTextContent();
            }
        }
        if (null == arch) {
            throw new BuildServiceException("Unable to determine architecture for repository \"" + repositoryName + "\"");
        }
        Architecture architecture = Architecture.parseArchitecture(arch);
        Repository repository = new Repository(repositoryName, architecture);
        for (Path path : parsePath(repositoryNode, architecture)) {
            repository.addPath(path);
        }
        return repository;
    }

    private static List<Path> parsePath(Node repositoryNode, Architecture arch) throws BuildServiceException {
        final NodeList childs = repositoryNode.getChildNodes();
        final List<Path> path = new ArrayList<Path>();
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("path".equals(node.getNodeName())) {
                path.add(new Path(Repository.getNewRepository(
                    Repositories.valueOf(node.getAttributes().getNamedItem("repository").getNodeValue()),
                    arch), new Project(node.getAttributes().getNamedItem("project").getNodeValue())));
            }
        }
        return path;
    }
}
